// $Header: /opt/cvs/java/net/logn/penrose/Constellation.java,v 1.11 2001/02/08 21:51:05 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.penrose;

// shapes for drawing
import java.awt.Shape;
// To provide updates to the screen
import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.awt.geom.GeneralPath;
// Tranformations to map cannonical patterns to real instances
import java.awt.geom.AffineTransform;
// Collections of objects
import java.util.Collection;
import java.util.Vector;
import java.util.SortedSet;
import java.util.HashSet;
import java.util.Iterator;


/**
 * <p>
 * A <b>Constellation</b> describes a configuration of points that form
 * a particular shape.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.11 $
 *
 * Last Modified $Date: 2001/02/08 21:51:05 $ by $Author: jhealy $
 */
public abstract class Constellation {

    /** Transformation that maps this Constellation to a real instance. */
    protected AffineTransform mapping;

    /** An empty shape to return when getShape() is called, for shapes
	that do not need to be drawn */
    protected static final Shape empty = new java.awt.geom.Rectangle2D.Double();

    /**
     * <p>
     * Returns the key pair of points in this constellation.
     * </p>
     *
     * @return IntersectionPoint[] The two points that make up the key
     * to this Constellation
     */
    public abstract IntersectionPoint[] getKeyPair();


    /**
     * <p>
     * Returns the pattern that marks the required intersections for this
     * constellation.
     * </p>
     *
     * @return IntersectionPoint[] The pattern defining this constellation
     */
    public abstract IntersectionPoint[] getPattern();


    /**
     * <p>
     * Returns the distance between the key pair in this constellation.
     * </p>
     *
     * @return double The distance between the two key pair points
     */
    public abstract double getDelta();


    /**
     * <p>
     * Returns the Tile to be drawn over this Constellation.
     * </p>
     *
     * @return Shape The Penrose Tile to be drawn over this Constellation
     */
    protected abstract Shape getTile();


    /**
     * <p>
     * Returns the Rhombus to be drawn over this Constellation.
     * </p>
     *
     * @return Shape The Penrose Rhomb to be drawn over this Constellation
     */
    protected abstract Shape getRhomb();


    /**
     * <p>
     * Returns the Shape to be drawn over this Constellation.
     * </p>
     *
     * @return Shape The tile or rhomb to be drawn over this Constellation
     */
    public Shape getShape() {
	if (PenroseTiling.tilesNotRhombs) {
	    return getTile();
	}
	else {
	    return getRhomb();
	}
    }


    /**
     * <p>
     * Draws a preview of the tile to the screen, to demonstrate that it has
     * been found.
     * </p>
     *
     * @param canvas The canvas to draw the preview to
     * @param path The path to draw
     */
    protected void drawPreview(Graphics2D canvas, GeneralPath path) {

	GeneralPath temp = (GeneralPath)path.clone();
	temp.transform(mapping);
	canvas.fill(temp);
    }


    /**
     * <p>
     * Draws a preview of the tile to the screen, to demonstrate that it has
     * been found.
     * </p>
     *
     * @param canvas The canvas to draw the preview to
     */
    protected void drawPreview(Graphics2D canvas) {
	canvas.fill(getShape());
    }


    /**
     * <p>
     * Returns all pairs of points that are the correct distance apart to
     * be the start of this constellation.
     * </p>
     *
     * @param pointSet The set of points to look for pairs in
     *
     * @return PointGraph All the point pairs that are at the correct
     * distance.
     */
    public abstract PointGraph scanForPairs(SortedSet pointSet);


    /**
     * <p>
     * Returns all pairs of points that are the correct distance apart.
     * </p>
     *
     * @param pointSet The set of points to look for pairs in
     * @param delta The distance between the points in the key pair
     *
     * @return PointGraph All the point pairs that are at the correct
     * distance
     */
    protected static PointGraph scanForPairs(SortedSet pointSet, double delta) {

	// Vector to hold all the pairs
	PointGraph pointPairs = new PointGraph();

	Object[] skyPoints = pointSet.toArray();

	for (int dammit = 0; dammit < skyPoints.length; dammit++) {

//	    System.out.println("SkyPoint " + ((IntersectionPoint)skyPoints[dammit]).toString());
	}

	for (int i = 0; i < (skyPoints.length - 1); i++) {
	    
	    for (int j = i+1; j < skyPoints.length; j++) {

		// Must declare points inside of for loop, or you'll end
		// up clobbering the references to pairs you've already
		// added to the vector!
		IntersectionPoint primary, secondary;
		
		primary = (IntersectionPoint)skyPoints[i];

		secondary = (IntersectionPoint)skyPoints[j];

		// now compare the points

		    //System.out.print("Considering " +
		//		       "\t" + primary + "\t" + secondary);
		if (PenroseTiling.inRange(primary.distance(secondary), delta)) {
		    //System.out.println("   Added!");
		    pointPairs.add(primary, secondary);
		}
		else {
		    //System.out.println("   Not!");
		}
	    }
	}

	return pointPairs;
    }
    


    /**
     * <p>
     * Returns a Constellation based on the pair of points passed in, if they
     * form the basis of a Constellation.
     * If the point pair does not turn out to be part of a match, null is
     * returned.
     * </p>
     *
     * @param pointSet The set of all possible points
     * @param plane The set of MusicalSequences
     * @param pair The pair of points to test
     *
     * @return Constellation A Constellation based on the point pair, or 
     * null if the point pair is not part of a match
     */
    public abstract Constellation testPair(SortedSet pointSet,
					   FiveFold plane,
					   IntersectionPoint[] pair);


    /**
     * <p>
     * Returns a transformation that maps this shape to the specified point
     * pair, if the point pair is really part of a copy of this Constellation.
     * If the point pair does not turn out to be part of a match, null is
     * returned.
     * </p>
     *
     * @param pointSet The set of all possible points
     * @param plane The set of MusicalSequences
     * @param pair The pair of points to test
     * @param keyPair The cannonical pair of points to check against
     * @param pattern The list of required points to check
     *
     * @return AffineTransform A transformation mapping the Constellation to
     * the point pair, or null if the point pair is not part of a match
     */
    protected static AffineTransform testRequiredBars(SortedSet pointSet,
						      FiveFold plane,
						      IntersectionPoint[] pair,
						      IntersectionPoint[] keyPair,
						      IntersectionPoint[] pattern) {
	

	// Construct a mapping between the two points
	AffineTransform map = getTransform(keyPair, pair);

	// if no mapping could be found, then stop now
	if (map == null) {
	    return null;
	}

	// Now, iterate over the template constellation.  Map each point 
	// with the map we just created, and then look for the point in 
	// our sky.  If it's there, keep going.  Otherwise, stop, and 
	// return false.

	IntersectionPoint mapped, unmapped;

	for (int i = 0; i < pattern.length; i++) {

	    unmapped = pattern[i];
	    mapped = plane.getIntersectionPoint(map.transform(unmapped, null));

	    //	    System.err.println("Mapped: " + unmapped + "\n" +
	    //			       "To    : " + map.transform(unmapped, null));
	    
	    // Moment of truth: look for the point in the sky
	    if ( (mapped == null) || (!pointSet.contains(mapped))) {

		//		System.err.println("Couldn't find in sky");
		// we failed to find a match
		return null;
	    }
	    else {
		// One last thing to check: make sure the relationship
		// between the Musical Sequences is the same in both
		// intersection points.  We do this because some patterns
		// can match incorrectly if they are reversed.
		
		double realDiff = unmapped.seq2.rotation - unmapped.seq1.rotation;
		double testDiff = mapped.seq2.rotation - mapped.seq1.rotation;
		
		if ( (realDiff != testDiff) &&
		     ( (realDiff + testDiff) != 360) ) {

		    //		    System.err.println("Rotations were wrong");
		    return null;
		}
	    }

	}

	// If we didn't exit above, then all of our points must have
	// matched.  Return the mapping.
	return map;
	
    }


    /**
     * <p>
     * Returns a collection of all known Constellations that have been found in
     * the set of points provided.
     * </p>
     *
     * @param pointSet The set of all possible points
     * @param plane The set of MusicalSequences
     * @param boundaries Vector of points that lie on the boundary
     * of a virtual box (or use null to disable boundaries)
     * @param canvas The canvas to show progress on
     *
     * @return Collection A Collection of Constellations that have been
     * found in the point set
     */
    public static Collection getAllConstellations(SortedSet pointSet,
						  FiveFold plane,
						  Collection boundaries,
						  Graphics2D canvas) {
	
	// use a hash set to prevent duplications
	HashSet constellations = new HashSet();
	
	constellations.addAll(Kite.getConstellations(pointSet,
						     plane,
						     boundaries,
						     canvas));

	constellations.addAll(Dart.getConstellations(pointSet,
						     plane,
						     boundaries,
						     canvas));

	constellations.addAll(DoubleKite.getConstellations(pointSet,
							   plane,
							   boundaries,
							   canvas));

	return constellations;
    }


    /**
     * <p>
     * Returns a collection of Constellations that have been found in
     * the set of points provided.  The type of constellation to get
     * is passed in as a parameter.
     * </p>
     *
     * @param pointSet The set of all possible points
     * @param plane The set of MusicalSequences
     * @param boundaries Vector of points that lie on the boundary
     * of a virtual box (or use null to disable boundaries)
     * @param prototype The constellation object to search for
     * @param canvas The canvas to show progress on, or null to disable
     *
     * @return Collection A Collection of Constellations that have been
     * found in the point set
     */
    protected static Collection getConstellations(SortedSet pointSet,
						  FiveFold plane,
						  Collection boundaries,
						  Constellation prototype,
						  Graphics2D canvas) {

	double delta = prototype.getDelta();
	IntersectionPoint[] keyPair = prototype.getKeyPair();
	IntersectionPoint[] pattern = prototype.getPattern();

	// Final collection to hold the found constellations in
	Vector constellations = new Vector();

	// Hold all the point pairs that we find
	PointGraph pairs = new PointGraph();

	if ( (boundaries == null) || (boundaries.size() < 2) ) {
	    // only one box to check... just look at all points
	    pairs = scanForPairs(pointSet, delta);		
	}
	else {
	    
	    Iterator boxIterator = boundaries.iterator();
	    
	    IntersectionPoint current = (IntersectionPoint)boxIterator.next();
	    IntersectionPoint old;
	    
	    SortedSet slice;
	    
	    // get everything up to the first point
	    // (not needed, as the first point should signal a boundary
	    // change, which will be caught by the while loop)
	    //slice = pointSet.headSet(current);
	    //pairs.addAll(scanForPairs(slice, delta));
	    
	    // partition the set into boxes and check each box
	    while (boxIterator.hasNext()) {
		old = current;
		current = (IntersectionPoint)boxIterator.next();
		slice = pointSet.subSet(old, current);
		pairs.addAll(scanForPairs(slice, delta));
	    }

	    // get everything from the last boundary to the end
	    slice = pointSet.tailSet(current);
	    pairs.addAll(scanForPairs(slice, delta));
	    
	}

	if (canvas != null) {
	    // if the canvas has been defined, then we can draw
	    // previews.  If not, we'll skip it.
	    // We perform this check outside the loops to improve
	    // efficiency.  The code in both this if and the else are
	    // exactly the same, except that this one renders a preview.
	    
	    IntersectionPoint[] primaries = pairs.getPrimaries();
	    
	    for (int i = 0; i < primaries.length; i++) {
		
		IntersectionPoint[] secondaries = pairs.getSecondaries(primaries[i]);
		
		for (int j = 0; j < secondaries.length; j++) {
		    
		    IntersectionPoint[] pair = {primaries[i], secondaries[j]};
		    
		    //System.out.println("Pair Scan:\n\t" + pair[0] + "\n\t" + pair[1]);
		    
		    Constellation found;
		    // check both symmetries of matches

		    // forward
		    found = prototype.testPair(pointSet, plane, pair);
		    
		    if (found != null) {
			found.drawPreview(canvas);
			constellations.add(found);
		    }
		    else {
			// reverse direction
			
			IntersectionPoint reverse[] = new IntersectionPoint[2];
			reverse[0] = pair[1];
			reverse[1] = pair[0];

			found = prototype.testPair(pointSet, plane, reverse);
			
			if (found != null) {
			    found.drawPreview(canvas);
			    constellations.add(found);
			}
		    }
		}
	    }
	} // end of preview-drawing code
	else {
	    // do not draw previews
	    
	    IntersectionPoint[] primaries = pairs.getPrimaries();
	    
	    for (int i = 0; i < primaries.length; i++) {
		
		IntersectionPoint[] secondaries = pairs.getSecondaries(primaries[i]);
		
		for (int j = 0; j < secondaries.length; j++) {
		    
		    IntersectionPoint[] pair = {primaries[i], secondaries[j]};
		    
		    //	    System.out.println("Pair Scan:\n\t" + pair[0] + "\n\t" + pair[1] +
		    //			       "(length " + pair.length + ")");
		    
		    Constellation found;
		    // check both symmetries of matches

		    // forward
		    found = prototype.testPair(pointSet, plane, pair);
		    
		    if (found != null) {
			constellations.add(found);
		    }
		    else {
			// reverse direction
			
			IntersectionPoint reverse[] = new IntersectionPoint[2];
			reverse[0] = pair[1];
			reverse[1] = pair[0];

			found = prototype.testPair(pointSet, plane, reverse);
			
			if (found != null) {
			    constellations.add(found);
			}
		    }
		}
	    }
	    
	} // end of no preview segment
	
	return constellations;
	
    }
	

    /**
     * <p>
     * Constructs a translation-rotation matrix that maps the "real"
     * point pair to the "test" point pair.  If a mapping cannot be
     * created, then null is returned.
     * </p>
     *
     * @return AffineTransform The mapping from one point pair to the
     * other, or null if no such mapping exists
     */
    public static AffineTransform getTransform(IntersectionPoint[] real,
					       IntersectionPoint[] test) {

	AffineTransform map = new AffineTransform();

	// Translate so that the first points in each pair line up
	double x = test[0].getX() - real[0].getX();
	double y = test[0].getY() - real[0].getY();

	// Now, move both sets of points to the origin so we can compute
	// the rotation about the origin

	double realX = real[1].getX() - real[0].getX();
	double realY = real[1].getY() - real[0].getY();
	double testX = test[1].getX() - test[0].getX();
	double testY = test[1].getY() - test[0].getY();

	double realTheta = PenroseTiling.atan2(realX, realY);
	double testTheta = PenroseTiling.atan2(testX, testY);
	
	double theta = testTheta - realTheta;

	//	System.out.println("\n\nTranslating: " + real[0] + "\n             " +
	//			   real[1] + " (" + realTheta + ")\nTo         : " +
	//			   test[0] + "\n             " + test[1] + " (" +
	//			   testTheta + ")");

	//	System.out.println("\nGot " + x + ", " + y + "  Rotate " + theta);
	// You have to read these next three lines backwords:
	// we translate the real point to the origin (last line)
	// then rotate (middle line)
	// then translate out to the test coordinates (first line)

	map.translate(test[0].getX(), test[0].getY());
	map.rotate(Math.toRadians(theta)); // perhaps should be negative?
	map.translate(-real[0].getX(), -real[0].getY());

	// Now we should check the mapping to make sure that the scale
	// is correct.

	return map;
    }


    /**
     * <p>
     * Maps a cannonical pattern point in to real space.  If the point is
     * forced, then the intersection point is complete.  Otherwise, the
     * first sequence in the intersection is the sequence that <b>should</b>
     * be forced if this point is matched, and the second sequence is null
     * to signify that the intersection is not forced.
     * </p>
     *
     * @param point The IntersectionPoint to map
     * @param map The mapping to use
     * @param plane The set of MusicalSequences
     * @param amount The number of sequences higher than the required bar
     * (mod 5) that the optional bar is
     *
     * @return IntersectionPoint The mapped intersection point, which may
     * include null sequences as explained above
     */
    public static IntersectionPoint mapOptional(IntersectionPoint point,
						AffineTransform map,
						FiveFold plane,
						int amount) {

	Point2D mapped = map.transform(point, null);

	int i = 0;

	int first = -1;
	int second = -1;

	while ((i < 5) && (first == -1) ) {
	    if (plane.isForced(mapped, plane.sequences[i])) {
		first = i;
	    }

	    i++;
	}

	while ((i < 5) && (second == -1) ) {
	    if (plane.isForced(mapped, plane.sequences[i])) {
		second = i;
	    }

	    i++;
	}

	if ( (first != -1) && (second != -1) ) {
	    // we have two sequences, so return a complete point
	  
	    // get the bar numbers
	    long bar1 = plane.getBarNum(plane.sequences[first], mapped);
	    long bar2 = plane.getBarNum(plane.sequences[second], mapped);

	    return new IntersectionPoint(plane.sequences[first], bar1,
					 plane.sequences[second], bar2, mapped);
	}
	else if (first != -1) {
	    // only found one bar (the one that must be forced)
	    // return a half-empty Intersection point

	    // get the sequence with the correct rotation
	    int optional = (first + amount) % 5;

	    IntersectionPoint temp = new IntersectionPoint(mapped);
	    temp.seq1 = plane.sequences[optional];

	    return temp;

	}

	// default if we fail...
	System.err.println("Couldn't find required bar!!!");
	return null;


    }


    /**
     * <p>
     * Uses this pattern to force any bars that we now know
     * must be forced.  Returns true if any new bars were forced.
     * </p>
     *
     * @return boolean True, if any new bars were forced
     */
    public abstract boolean forceBars(FiveFold plane);


    /**
     * <p>
     * Prints out a string representation of this object.
     * </p>
     *
     * @return String A string representation of this object
     */
    public abstract String toString();
    

    public static void main(String[] args) {
	
	AffineTransform map = new AffineTransform();
	java.awt.geom.Point2D point = new java.awt.geom.Point2D.Double(3.0, 3.0);
	java.awt.geom.Point2D other = new java.awt.geom.Point2D.Double(1.0, 0.0);
	
	/*
	  map.translate(3.0, 3.0);
	  map.rotate(Math.toRadians(1));
	  map.translate(-3.0, -3.0);
	*/
	
	// scaleX shearX transX
	// shearY scaleY transY
	
	double theta = Math.toRadians(-45);
	
	map = new AffineTransform(Math.cos(theta), -Math.sin(theta), 0.0,
				  Math.sin(theta), Math.cos(theta), 0.0);
	
	System.out.println(map);
	System.out.println(map.transform(point, null));
	System.out.println(map.transform(other, null));
	
    }

}



