// $Header: /opt/cvs/java/net/logn/penrose/Dart.java,v 1.9 2001/02/08 21:51:05 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.penrose;

// shapes for drawing
import java.awt.Shape;
import java.awt.geom.GeneralPath;
// To provide updates to the screen
import java.awt.Graphics2D;
// Tranformations to map cannonical patterns to real instances
import java.awt.geom.AffineTransform;

// Collections of objects
import java.util.Vector;
import java.util.Collection;
import java.util.SortedSet;
import java.util.Iterator;


/**
 * <p>
 * This <b>Dart</b> class implements the methods for finding a Dart in a
 * Penrose Tiling.  Darts contain bars that are "optional", and so can
 * force new information when they are found.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.9 $
 *
 * Last Modified $Date: 2001/02/08 21:51:05 $ by $Author: jhealy $
 */
public class Dart extends Constellation {

    /** List of intersection points that make up this Constellation */
    protected static IntersectionPoint[] pattern;

    /** A pair of points that define the beginning of this Constellation */
    protected static IntersectionPoint[] keyPair = new IntersectionPoint[2];

    /** The distance between the two key points */
    protected static double delta;

    /** The shape of a Dart as a Tile */
    protected static GeneralPath tile;

    /** The shape of a Dart as a Rhomb */
    protected static GeneralPath rhomb;

    /** Left-side intersection point that is optional */
    protected static IntersectionPoint optionalLeft;

    /** Right-side intersection point that is optional */
    protected static IntersectionPoint optionalRight;

    /** Left-side intersection point */
    private IntersectionPoint left;

    /** Right-side intersection point */
    private IntersectionPoint right;


    static {
	// initialize the static class members

	// Set up the intersections
	FiveFold plane = PenroseTiling.aceConfiguration();

	pattern = new IntersectionPoint[3];
	
	// now get all the intersections of all the bars
	pattern[0] = plane.getIntersectionPoint(plane.sequences[0], 0,
						plane.sequences[2], 0);
	pattern[1] = plane.getIntersectionPoint(plane.sequences[0], 0,
						plane.sequences[3], 0);
	pattern[2] = plane.getIntersectionPoint(plane.sequences[2], 0,
						plane.sequences[3], 0);

	
	keyPair[0] = pattern[0];
	keyPair[1] = pattern[2];

	delta = keyPair[0].distance(keyPair[1]);

	// load the "optional" intersections
	optionalLeft = plane.getIntersectionPoint(plane.sequences[0], 0,
						  plane.sequences[4], 0);

	optionalRight = plane.getIntersectionPoint(plane.sequences[0], 0,
						   plane.sequences[1], 0);

	// Define the tile shape
	tile = new GeneralPath();
	tile.moveTo(0.0f, 0.0f);

	float cornerX = (float)(PenroseTiling.cos(36) *
	    (PenroseTiling.MINNICK_X + PenroseTiling.MINNICK_Y));
	float cornerY = (float)(PenroseTiling.sin(36) *
	    (PenroseTiling.MINNICK_X + PenroseTiling.MINNICK_Y));

	// Minnick A is close, but not quite right (check the diagrams)
	//	float topX = (float)(PenroseTiling.MINNICK_A);

	float topX = cornerX - (float)(PenroseTiling.cos(72) *
				       (PenroseTiling.MINNICK_W +
					PenroseTiling.MINNICK_Z));
	
	tile.lineTo(cornerX, cornerY);
	tile.lineTo(topX, 0.0f);
	tile.lineTo(cornerX, -cornerY);

	tile.closePath();

	// Define the rhomb shape
	// (for now, just draw the dart triangle...)
	rhomb = new GeneralPath();
	rhomb.moveTo(0.0f, 0.0f);

	float cosTau = (float)(PenroseTiling.cos(36) * PenroseTiling.TAU);
	float sinTau = (float)(PenroseTiling.sin(36) * PenroseTiling.TAU);
	
	rhomb.lineTo(cosTau, sinTau);
	rhomb.lineTo((2 * cosTau), 0.0f);
	rhomb.lineTo(cosTau, -sinTau);
	
	rhomb.closePath();

    }


    /**
     * <p>
     * Constructor.  Generates the "cannonical" Dart.
     * </p>
     */
    public Dart() {

	this(new AffineTransform());

    }
    

    /**
     * <p>
     * Constructor.  Generates a dart mapped into the position specified
     * by the transformation provided.
     * </p>
     *
     * @param map The transformation mapping the Dart to a position in
     * the plane
     */
    public Dart(AffineTransform map) {

	this(map, null, null);

    }
    

    /**
     * <p>
     * Constructor.  Generates a dart mapped into the position specified
     * by the transformation provided.
     * </p>
     *
     * @param map The transformation mapping the Dart to a position in
     * the plane
     * @param leftPoint The optional left point, or null if it is not forced
     * @param rightPoint The optional right point, or null if it is not forced
     */
    public Dart(AffineTransform map, IntersectionPoint leftPoint,
		IntersectionPoint rightPoint) {

	mapping = map;
	left = leftPoint;
	right = rightPoint;

    }
    

    /**
     * <p>
     * Returns the key pair of points in this constellation.
     * </p>
     *
     * @return IntersectionPoint[] The two points that make up the key
     * to this Constellation
     */
    public IntersectionPoint[] getKeyPair() {

	return keyPair;
    }


    /**
     * <p>
     * Returns the pattern that marks the required intersections for this
     * constellation.
     * </p>
     *
     * @return IntersectionPoint[] The pattern defining this constellation
     */
    public IntersectionPoint[] getPattern() {

	return pattern;
    }


    /**
     * <p>
     * Returns the distance between the key pair in this constellation.
     * </p>
     *
     * @return double The distance between the two key pair points
     */
    public double getDelta() {

	return delta;

    }


    /**
     * <p>
     * Returns the Tile to be drawn over this Constellation.
     * </p>
     *
     * @return Shape The Penrose Tile to be drawn over this Constellation
     */
    protected Shape getTile() {
	GeneralPath temp = (GeneralPath)tile.clone();
	temp.transform(mapping);
	return temp;
    }


    /**
     * <p>
     * Returns the Rhombus to be drawn over this Constellation.
     * </p>
     *
     * @return Shape The Penrose Rhomb to be drawn over this Constellation
     */
    protected Shape getRhomb() {
	GeneralPath temp = (GeneralPath)rhomb.clone();
	temp.transform(mapping);
	return temp;
    }


    /**
     * <p>
     * Returns all pairs of points that are the correct distance apart to
     * be the start of this constellation.
     * </p>
     *
     * @param pointSet The set of points to look for pairs in
     *
     * @return PointGraph All the point pairs that are at the correct
     * distance.
     */
    public PointGraph scanForPairs(SortedSet pointSet) {

	return scanForPairs(pointSet, delta);
    }
    

    /**
     * <p>
     * Returns a Constellation based on the pair of points passed in, if they
     * form the basis of a Constellation.
     * If the point pair does not turn out to be part of a match, null is
     * returned.
     * </p>
     *
     * @param pointSet The set of all possible points
     * @param plane The set of MusicalSequences
     * @param pair The pair of points to test
     *
     * @return Constellation A Constellation based on the point pair, or 
     * null if the point pair is not part of a match
     */
    public Constellation testPair(SortedSet pointSet,
				  FiveFold plane,
				  IntersectionPoint[] pair) {
	
	//	System.err.println("\nChecking:");
	//	System.err.println("\t" + pair[0]);
	//	System.err.println("\t" + pair[1]);

	AffineTransform map = testRequiredBars(pointSet, plane, pair,
					       keyPair, pattern);

	// if we couldn't find the required points, then just return null
	// (no need to search for optional bars...)
	if (map == null) {
	    //	    System.err.println("Failed Required Bars");
	    return null;
	}

	// otherwise, we should now search for the optional bars
	IntersectionPoint l = mapOptional(optionalLeft, map, plane, 4);
	IntersectionPoint r = mapOptional(optionalRight, map, plane, 1);

	if ( (l.seq2 != null) || (r.seq2 != null) ) {
	    return new Dart(map, l, r);
	}
	
	// otherwise, return null
	//	System.err.println("Failed Optional Bars");
	return null;
	
    }


    /**
     * <p>
     * Returns a collection of all Constellations that have been found in
     * the set of points provided.
     * </p>
     *
     * @param pointSet The set of all possible points
     * @param plane The set of MusicalSequences
     * @param boundaries Vector of points that lie on the boundary
     * of a virtual box (or use null to disable boundaries)
     * @param canvas The canvas to show progress on
     *
     * @return Collection A Collection of Constellations that have been
     * found in the point set
     */
    public static Collection getConstellations(SortedSet pointSet,
					       FiveFold plane,
					       Collection boundaries,
					       Graphics2D canvas) {

	return getConstellations(pointSet, plane, boundaries,
				 (new Dart()), canvas);
	
    }


    /**
     * <p>
     * Uses this pattern to force any bars that we now know
     * must be forced.  Returns true if any new bars were forced.
     * </p>
     *
     * @return boolean True, if any new bars were forced
     */
    public boolean forceBars(FiveFold plane) {

	if (left.seq2 == null) {	    
	    return plane.forcePoint(left.seq1, left);
	}
	else if (right.seq2 == null) {
	    return plane.forcePoint(right.seq1, right);
	}
	else {
	    return false;
	}

    }


    /**
     * <p>
     * Prints out a string representation of this object.
     * </p>
     *
     * @return String A string representation of this object
     */
    public String toString() {

	return "Dart with " + mapping;
    }
    

    /**
     * <p>
     * Tests the class
     * </p>
     *
     * @param args Command-line arguments
     */
    public static void main(String[] args) {

	FiveFold fiveFold = PenroseTiling.sunConfiguration();

	System.out.println(":::Template points");
	System.out.println(":::\t" + pattern[0] + "\n:::\t" +
			   pattern[1] + "\n:::\t" + pattern[2]);

	SortedSet points = fiveFold.findIntersectionPoints(-0.7, -0.5,
							   0.01, 0.01);

	for (Iterator skyIterator = points.iterator();
	     skyIterator.hasNext();
	     ) {
	    
	    System.out.println(":::Sky " + skyIterator.next());
	}


	// find patterns
	Collection darts = Dart.getConstellations(points, fiveFold, null, null);

	for (Iterator dartIterator = darts.iterator();
	     dartIterator.hasNext();
	     ) {
	    
	    Dart dart = (Dart)dartIterator.next();
	    System.out.println(":::Found " + dart);
	}
	
	
    }
    

}


