// $Header: /opt/cvs/java/net/logn/penrose/DoubleKite.java,v 1.2 2001/02/08 21:51:05 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.penrose;

// shapes for drawing
import java.awt.Shape;
import java.awt.geom.GeneralPath;
// To provide updates to the screen
import java.awt.Graphics2D;
// Tranformations to map cannonical patterns to real instances
import java.awt.geom.AffineTransform;
// Collections of objects
import java.util.Vector;
import java.util.Collection;
import java.util.SortedSet;
import java.util.Iterator;


/**
 * <p>
 * This <b>DoubleKite</b> class implements the methods for finding the
 * so-called "double kite" formation, as described in Minnick[1998].
 * This constellation does not need to be drawn (it is formed by Kites,
 * which are found separately).  However, the DoubleKite does force new
 * Ammann Bars when found, and thus expands our empires.  When found, 
 * a DoubleKite should be used to force new bars.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.2 $
 *
 * Last Modified $Date: 2001/02/08 21:51:05 $ by $Author: jhealy $
 */
public class DoubleKite extends Constellation {

    /** List of intersection points that make up this Constellation */
    protected static IntersectionPoint[] pattern;

    /** A pair of points that define the beginning of this Constellation */
    protected static IntersectionPoint[] keyPair = new IntersectionPoint[2];

    /** A point that shows where to force a new bar when a double kite
	is found */
    protected static IntersectionPoint force;

    /** The distance between the two key points */
    protected static double delta;

    /** The shape of a Kite as a Tile */
    protected static GeneralPath tile;

    /** The shape of a Kite as a Rhomb */
    protected static GeneralPath rhomb;


    static {
	// initialize the static class members

	// Set up the intersections
	FiveFold plane = PenroseTiling.deuceConfiguration();

	pattern = new IntersectionPoint[5];
	
	// now get all the intersections of all the bars
	pattern[0] = plane.getIntersectionPoint(plane.sequences[0], 0,
						plane.sequences[1], 0);
	pattern[1] = plane.getIntersectionPoint(plane.sequences[0], 0,
						plane.sequences[2], 0);
	pattern[2] = plane.getIntersectionPoint(plane.sequences[0], 0,
						plane.sequences[4], 0);
	pattern[3] = plane.getIntersectionPoint(plane.sequences[2], 0,
						plane.sequences[4], 0);
	pattern[4] = plane.getIntersectionPoint(plane.sequences[3], 0,
						plane.sequences[4], 0);

	
	keyPair[0] = pattern[1];
	keyPair[1] = pattern[3];

	delta = keyPair[0].distance(keyPair[1]);

	// force the bar that should appear in the doublekite formation
	plane.sequences[2].force(-1, PenroseTiling.SHORTER);

	// get the intersection point that represents that bar
	force = plane.getIntersectionPoint(plane.sequences[2], -1,
					   plane.sequences[1], 0);

	// Define the preview shape

	float centerX = (float)(PenroseTiling.cos(36) * 
				(PenroseTiling.MINNICK_X +
				 PenroseTiling.MINNICK_Y));
	
	float centerY = (float)-(PenroseTiling.sin(36) * 
				 (PenroseTiling.MINNICK_X +
				  PenroseTiling.MINNICK_Y));

	float topX = (float)(PenroseTiling.cos(36) * 
			     (PenroseTiling.MINNICK_X +
			      PenroseTiling.MINNICK_Y +
			      PenroseTiling.MINNICK_W +
			      PenroseTiling.MINNICK_Z));
	
	float topY = (float)-(PenroseTiling.sin(36) * 
			      (PenroseTiling.MINNICK_X +
			      PenroseTiling.MINNICK_Y +
			      PenroseTiling.MINNICK_W +
			      PenroseTiling.MINNICK_Z));
	
	// This side distance isn't perfect, but it's close enough for a preview
	double sideDistance = 2 * (PenroseTiling.cos(36) * PenroseTiling.TAU);
	
	// Define the tile
	tile = new GeneralPath();
	tile.moveTo(centerX, centerY);

	tile.lineTo((float)sideDistance, 0.0f);

	tile.lineTo((float)(PenroseTiling.cos(288) * sideDistance),
		    (float)(PenroseTiling.sin(288) * sideDistance) );

	tile.closePath();

	// define the rhomb
	rhomb = new GeneralPath();
	rhomb.moveTo(centerX, centerY);

	rhomb.lineTo((float)sideDistance, 0.0f);

	rhomb.lineTo(topX, topY);

	rhomb.lineTo((float)(PenroseTiling.cos(288) * sideDistance),
		    (float)(PenroseTiling.sin(288) * sideDistance) );

	rhomb.closePath();

    }


    /**
     * <p>
     * Constructor.  Generates the "cannonical" DoubleKite.
     * </p>
     */
    public DoubleKite() {

	this(new AffineTransform());

    }
    

    /**
     * <p>
     * Constructor.  Generates a double kite mapped into the position specified
     * by the transformation provided.
     * </p>
     *
     * @param map The transformation mapping the DoubleKite to a position in
     * the plane
     */
    public DoubleKite(AffineTransform map) {

	mapping = map;

    }


    /**
     * <p>
     * Returns the key pair of points in this constellation.
     * </p>
     *
     * @return IntersectionPoint[] The two points that make up the key
     * to this Constellation
     */
    public IntersectionPoint[] getKeyPair() {

	return keyPair;
    }


    /**
     * <p>
     * Returns the pattern that marks the required intersections for this
     * constellation.
     * </p>
     *
     * @return IntersectionPoint[] The pattern defining this constellation
     */
    public IntersectionPoint[] getPattern() {

	return pattern;
    }


    /**
     * <p>
     * Returns the distance between the key pair in this constellation.
     * </p>
     *
     * @return double The distance between the two key pair points
     */
    public double getDelta() {

	return delta;

    }


    /**
     * <p>
     * Returns an empty shape (the double kite is not explicitly drawn)
     * </p>
     *
     * @return Shape An empty shape
     */
    protected Shape getTile() {
	return empty;
    }


    /**
     * <p>
     * Returns an empty shape (the double kite is not explicitly drawn)
     * </p>
     *
     * @return Shape An empty shape
     */
    protected Shape getRhomb() {
	GeneralPath temp = (GeneralPath)rhomb.clone();
	temp.transform(mapping);
	return temp;
    }


    /**
     * <p>
     * Draws a preview of the tile to the screen, to demonstrate that it has
     * been found.
     * </p>
     *
     * @param canvas The canvas to draw the preview to
     */
    protected void drawPreview(Graphics2D canvas) {
	if (PenroseTiling.tilesNotRhombs) {
	    drawPreview(canvas, tile);
	}
	else {
	    drawPreview(canvas, rhomb);
	}
    }


    /**
     * <p>
     * Returns all pairs of points that are the correct distance apart to
     * be the start of this constellation.
     * </p>
     *
     * @param pointSet The set of points to look for pairs in
     *
     * @return PointGraph All the point pairs that are at the correct
     * distance.
     */
    public PointGraph scanForPairs(SortedSet pointSet) {

	return scanForPairs(pointSet, delta);
    }
    

    /**
     * <p>
     * Returns a Constellation based on the pair of points passed in, if they
     * form the basis of a Constellation.
     * If the point pair does not turn out to be part of a match, null is
     * returned.
     * </p>
     *
     * @param pointSet The set of all possible points
     * @param plane The set of MusicalSequences
     * @param pair The pair of points to test
     *
     * @return Constellation A Constellation based on the point pair, or 
     * null if the point pair is not part of a match
     */
    public Constellation testPair(SortedSet pointSet,
				  FiveFold plane,
				  IntersectionPoint[] pair) {
	
	AffineTransform map = testRequiredBars(pointSet, plane, pair,
					       keyPair, pattern);
	
	if (map != null) {
	    return new DoubleKite(map);
	}
	
	// no match -- give up
	return null;
	
    }


    /**
     * <p>
     * Returns a collection of all Constellations that have been found in
     * the set of points provided.
     * </p>
     *
     * @param pointSet The set of all possible points
     * @param plane The set of MusicalSequences
     * @param boundaries Vector of points that lie on the boundary
     * of a virtual box (or use null to disable boundaries)
     * @param canvas The canvas to show progress on
     *
     * @return Collection A Collection of Constellations that have been
     * found in the point set
     */
    public static Collection getConstellations(SortedSet pointSet,
					       FiveFold plane,
					       Collection boundaries,
					       Graphics2D canvas) {

	return getConstellations(pointSet, plane, boundaries,
				 (new DoubleKite()), canvas);
	
    }


    /**
     * <p>
     * Uses this pattern to force any bars that we now know
     * must be forced.  Returns true if any new bars were forced.
     * </p>
     *
     * @return boolean True, if any new bars were forced
     */
    public boolean forceBars(FiveFold plane) {


	// map the forced point	
	IntersectionPoint newPoint = mapOptional(force, mapping, plane, 1);

	if (newPoint.seq2 == null) {
	    return plane.forcePoint(newPoint.seq1, newPoint);
	}
	else {
	    return false;
	}

    }


    /**
     * <p>
     * Prints out a string representation of this object.
     * </p>
     *
     * @return String A string representation of this object
     */
    public String toString() {

	return "DoubleKite with " + mapping;
    }
    

    /**
     * <p>
     * Tests the class
     * </p>
     *
     * @param args Command-line arguments
     */
    public static void main(String[] args) {

	FiveFold fiveFold = PenroseTiling.sunConfiguration();

	System.out.println(":::Template points");
	System.out.println(":::\t" + pattern[0] + "\n:::\t" +
			   pattern[1] + "\n:::\t" + pattern[2]);

	SortedSet points = fiveFold.findIntersectionPoints(-0.7, -0.5,
							   0.01, 0.01);

	for (Iterator skyIterator = points.iterator();
	     skyIterator.hasNext();
	     ) {
	    
	    System.out.println(":::Sky " + skyIterator.next());
	}


	// find patterns
	Collection doubleKites = DoubleKite.getConstellations(points, fiveFold, null, null);

	for (Iterator doubleKiteIterator = doubleKites.iterator();
	     doubleKiteIterator.hasNext();
	     ) {
	    
	    DoubleKite doubleKite = (DoubleKite)doubleKiteIterator.next();
	    System.out.println(":::Found " + doubleKite);
	}
	
	
    }
    

}


