// $Header: /opt/cvs/java/net/logn/penrose/FiveFold.java,v 1.9 2001/02/07 20:08:45 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.penrose;

// Used to mark where the sequences originate
import java.awt.geom.Point2D;
// Allows us to define areas in the plane
import java.awt.geom.Rectangle2D;
// Allow us to draw the sequences
import java.awt.geom.Line2D;
// Allows us to store objects
import java.util.SortedSet;
import java.util.TreeSet;
// Allows us to save the object out to a file
import java.io.Serializable;


/**
 * <p>
 * A <b>Five Fold</b> represents the interaction of five Musical Sequences
 * in the plane.  The sequences are separated by multiples of 72 degrees
 * from each other.
 * </p>
 *
 * <p>
 * The goal of the Five Fold is to allow us to find the intersections between
 * Musical Sequences and look for patterns.  These patterns will tell us where
 * Penrose Tiles are forced.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.9 $
 *
 * Last Modified $Date: 2001/02/07 20:08:45 $ by $Author: jhealy $
 */
public class FiveFold implements Serializable {
    
    /** The number of sequences in the plane */
    public static final int NUMSEQUENCES = 5;

    /** The collection of five Musical Sequences */
    protected MusicalSequence sequences[];
    

    /**
     * <p>
     * Creates a new plane with 5 interacting musical sequences.
     * </p>
     */
    public FiveFold() {
	
	sequences = new MusicalSequence[NUMSEQUENCES];
	
	for (int i = 0; i < NUMSEQUENCES; i++)	{
	    sequences[i] = new MusicalSequence(0,0,(i * 72));
	}
	
    }
    
    
    /**
     * <p>
     * Returns the sequence with the given angle.
     * </p>
     *
     * @param angle The angle of the sequence we would like
     *
     * @return MusicalSequence The sequence with the specified angle
     */
    public MusicalSequence getSequence(double angle) {

	// Since we know what angles we gave to the
	// sequences, we can just use a little math:
	  
	int index = (int)(Math.rint((angle / 72)));

	if (index < NUMSEQUENCES) {
	    return sequences[index];
	}
	else {
	    return null;
	}
	
    }
    
        
    /**
     * <p>
     * Returns the sequence whose bar contains the point p.
     * </p>
     *
     * @param p The point to find the sequence for
     *
     * @return MusicalSequence The sequence that contains the point.
     * Returns null if the point does not lie on <b>exactly one</b> sequence.
     */
    public MusicalSequence getSequence(Point2D p) {
	
	int seq = -1;
	int count = 0;

	for (int i = 0; i < NUMSEQUENCES; i++) {
	    if (isForced(p, sequences[i])) {
		count++;
		seq = i;
	    }
	}
	
	if (count == 1) {
	    return sequences[seq];
	}
	else {
	    if (count > 1) {
		System.err.println("Too many bars!");
	    }
	    return null;
	}
	
    }
    

    /**
     * <p>
     * Forces a bar in the given sequence to run through the specified
     * point.
     * </p>
     *
     * @param ms The MusicalSequence to force
     * @param p The point to force through
     *
     * @return boolean True, if the bar was not already forced
     */
    public static boolean forcePoint(MusicalSequence ms, Point2D p) {

	Point2D alongAxis = findNearestPoint(p, ms);

	double distance = getDistanceAlongSequence(alongAxis, ms);

	//System.err.println("Forcing at " + distance + " along " + ms.rotation);

	// Now, just force a bar at that distance
	return ms.forceAtDistance(distance);

    }


    /**
     * <p>
     * Returns the distance from the origin to the given point.
     * </p>
     *
     * @param p The point to measure the distance to
     *
     * @return double The distance from (0, 0) to that point
     */
    public static double originToPoint(Point2D p) {
	
	Point2D zero = new Point2D.Double(0.0, 0.0);

	return zero.distance(p);
    }
    
    
    /**
     * <p>
     * Returns the distance from this point to the origin.
     * </p>
     *
     * @param x The x coordinate
     * @param y The y coordinate
     *
     * @return double The distance to the given point from the origin
     */
     public static double originToPoint(double x, double y) {
	
	 Point2D temp = new Point2D.Double(x, y);

	 return originToPoint(temp);
	
    }
    
    
    /**
     * <p>
     * Returns the distance along the axis of the Musical Sequence.  This
     * is similar to a vector dot product; the distance is the component
     * that lies along the axis of the sequence.
     * </p>
     *
     * @param x The x coordinate of the point to measure the distance to
     * @param y The y coordinate of the point to measure the distance to
     * @param ms The Musical Sequence to measure the distance along
     *
     * @return double The component of the distance that lies along the
     * sequence axis
     */
    public static double getDistanceAlongSequence(double x, double y,
						  MusicalSequence ms) {

	return getDistanceAlongSequence(new Point2D.Double(x, y), ms);
	
    }
    

    /**
     * <p>
     * Returns the distance along the axis of the Musical Sequence.  This
     * is similar to a vector dot product; the distance is the component
     * that lies along the axis of the sequence.
     * </p>
     *
     * @param p The point to measure the distance to
     * @param ms The Musical Sequence to measure the distance along
     *
     * @return double The component of the distance that lies along the
     * sequence axis
     */
    public static double getDistanceAlongSequence(Point2D p, MusicalSequence ms) {
	
	Point2D center = new Point2D.Double(ms.centerX, ms.centerY);

	// Just get the distance between the center and the point
	double distance = center.distance(p);

	// Now get the rotation between the center and the point
	double theta = PenroseTiling.atan2( (p.getX() - ms.centerX),
					    (p.getY() - ms.centerY));

	//	System.err.println("Point: " + theta + " Sequence " + ms.rotation);

	// If the rotation is the sequence's rotation, then the distance
	// is positive
	if (PenroseTiling.inRange(theta, ms.rotation)) {
	    return distance;
	}
	else {
	    // otherwise, it's negative (rotation + 180)
	    return -distance;
	}

    }
    

    /**
     * <p>
     * Returns the point given after being translated parallel to the axis
     * of the given sequence.
     * </p>
     *
     * @param p The point being translated
     * @param ms The sequence to translate parallel to
     * @param distance The distance to translate
     *
     * @return Point2D The translated point
     */
    protected static Point2D translateAlongSequence(Point2D p,
						    MusicalSequence ms,
						    double distance) {
	// Uses double version below

	return translateAlongSequence(p.getX(), p.getY(), ms, distance);
    }
    

    /**
     * <p>
     * Returns the point given after being translated parallel to the axis
     * of the given sequence.
     * </p>
     *
     * @param x The x coordinate of the point being translated
     * @param y The y coordinate of the point being translated
     * @param ms The sequence to translate parallel to
     * @param distance The distance to translate
     *
     * @return Point2D The translated point
     */
    protected static Point2D translateAlongSequence(double x, double y,
						    MusicalSequence ms,
						    double distance) {
	
	double angle = ms.rotation;
	double xOff = (distance * PenroseTiling.cos(angle));
	double yOff = (distance * PenroseTiling.sin(angle));
	
	Point2D newPoint = new Point2D.Double((x + xOff), (y + yOff));
	
	return newPoint;
    }
    
    
    /**
     * <p>
     * Returns a point that lies on the axis of the given sequence, at
     * the bar number specified.
     * </p>
     *
     * @param ms The sequence to follow
     * @param barNum The bar number of the sequence to use
     *
     * @return Point2D The intersection of the axis and bar number
     */
    protected static Point2D barNumToPoint(MusicalSequence ms, long barNum) {
	// Uses distance to point and the musical sequence's built-in
	// Conversion from a bar number to a distance
	
	return distanceToPoint(ms, ms.barToDouble(barNum));
	
    }
    
    
    /**
     * <p>
     * Returns a point that lies along the given sequence's axis at the
     * distance specified.
     * </p>
     *
     * @param ms The sequence to move along
     * @param distance The distance along the sequence to look
     *
     * @return Point2D The point that lies along the axis by amount given
     */
    protected static Point2D distanceToPoint(MusicalSequence ms, double distance) {

	double angle = ms.rotation;
	double x;
	double y;
	
	x = distance * PenroseTiling.cos(angle);
	y = distance * PenroseTiling.sin(angle);
	
	x += ms.centerX;
	y += ms.centerY;
	
	return new Point2D.Double(x,y);
    }
    

    /**
     * <p>
     * Returns the bar number on the given sequence that is nearest to
     * this point.
     * </p>
     *
     * @param ms The sequence to check against
     * @param p The point to find a bar for
     *
     * @return long The bar number on the sequence that is closest to the point
     */
    protected static long getBarNum(MusicalSequence ms, Point2D p) {

	Point2D seqPoint = findNearestPoint(p, ms);
	    
	// We now have a Point2D that lies along the axis of our sequence.
	// Now we get the distance along the musical sequence

	double d = getDistanceAlongSequence(seqPoint, ms);

	// and convert that to a bar number

	long bar = ms.doubleToBar(d);

	return bar;
    }

    
    /**
     * <p>
     * Returns the intersection that occurs nearest this point.
     * </p>
     *
     * @param p The point where the intersection occurs
     *
     * @return IntersectionPoint The Ammann Bar intersection at this point
     */
    protected IntersectionPoint getIntersectionPoint(Point2D p) {
	
	int i = 0;

	int first = -1;
	int second = -1;

//System.err.println("\ngetIntersectionPoint: " + p + "\n");
	// find the sequences that intersect here

	while ((i < NUMSEQUENCES) && (first == -1) ) {
	    if (isForced(p, sequences[i])) {
		first = i;
	    }

//else{System.err.println("Sequence " + sequences[i].rotation + " isn't forced.");}
	    
	    i++;
	}

	while ((i < NUMSEQUENCES) && (second == -1) ) {
	    if (isForced(p, sequences[i])) {
		second = i;
	    }

//else{System.err.println("Sequence " + sequences[i].rotation + " isn't forced.");}
	    
	    i++;
	}

	if ( (first != -1) && (second != -1) ) {
	    // find the bars that are forced

	    long bar1 = getBarNum(sequences[first], p);
	    long bar2 = getBarNum(sequences[second], p);

//System.err.println("Found intersecting sequences " + first + " and " + second);
	    return new IntersectionPoint(sequences[first], bar1,
					 sequences[second], bar2, p);
	}
	else {
//System.err.println("While getting the intersection, I couldn't find two forced bars!");
//System.err.println(p.toString() + " Doesn't seem to be on a forced bar, and I think that the sequences are " + first + " and " + second);
	    return null;
	}

    }


    /**
     * <p>
     * Returns the intersection between the two sequence-bar combinations given.
     * </p>
     *
     * @param aRot The rotation of the first sequence
     * @param aBarNum The bar number on the first sequence
     * @param bRot The rotation of the second sequence
     * @param bBarNum The bar number on the second sequence
     *
     * @return IntersectionPoint The intersection between these two bars
     */
    public IntersectionPoint getIntersectionPoint(double aRot, long aBarNum,
						  double bRot, long bBarNum) {

	MusicalSequence a = getSequence(aRot);
	MusicalSequence b = getSequence(bRot);

	return getIntersectionPoint(a, aBarNum, b, bBarNum);
    }


    /**
     * <p>
     * Returns the intersection between the two sequence-bar combinations given.
     * </p>
     *
     * @param a The first sequence
     * @param aBarNum The bar number on the first sequence
     * @param b The second sequence
     * @param bBarNum The bar number on the second sequence
     *
     * @return IntersectionPoint The intersection between these two bars
     */
    public static IntersectionPoint getIntersectionPoint(MusicalSequence a,
							 long aBarNum,
							 MusicalSequence b,
							 long bBarNum) {
	
	// Get a Point2D on each line:
	
	Point2D aPoint = barNumToPoint(a, aBarNum);
	Point2D bPoint = barNumToPoint(b, bBarNum);
	
	// Use rotation+90, because the bars of the sequence are perpendicular
	// to the axis of the sequence.
	Point2D intersection = getIntersection(aPoint, (a.rotation + 90),
					       bPoint, (b.rotation + 90));

	return new IntersectionPoint(a, aBarNum, b, bBarNum, intersection);
    }
    
    
    /**
     * <p>
     * Returns the intersection point of the lines with the given rotations
     * intercepting the given points.  (That is, the lines defined by a
     * single point and the angle through them).
     * </p>
     *
     * @param aPoint A point that the first line goes through
     * @param aRot The rotation of the first line
     * @param bPoint A point that the second line goes through
     * @param bRot The roatation of the second line
     *
     * @return The intersection of the lines
     */
    public static Point2D getIntersection(Point2D aPoint, double aRot,
					  Point2D bPoint, double bRot) {

	// Sanity check:
	if (aRot == bRot) {
	    return null;	// If the rotation is the same, then the bars never intersect
	}
	
	//System.out.println("\nPoints are at:\n\t" + aPoint.toString() + "\n\t" + bPoint.toString());
	
	/*
	 *	Now, construct a line through that Point, which
	 *	represents the bar.  Note that the bars run
	 *	perpendicular to the direction of the sequence
	 * 
	 * A note about useless assignments:
	 * 
	 * Because Java wants to make sure that we aren't using
	 * unassigned values, we have to make useless assignments here
	 * so that java will compile this section.  I have provided
	 * comments to attepmt to prove that no matter how one traces
	 * through these if-then statements, it will all work out fine.
	 * 
	 */
	
	double xInt = Double.NaN;	// X-coordinate of the intersection of the lines
	double yInt = Double.NaN;	// Y-coordinate of the intersection of the lines
	double aSlope = Double.NaN; 	// The slope of the bar in sequence a
	double bSlope = Double.NaN;  	// The slope of the bar in sequence b
	double aInt = Double.NaN;	// The y-intercept of the bar in sequence a
	double bInt = Double.NaN;	// The y-intercept of the bar in sequence b
	
	if (aRot == 90)	{ // Horizontal case; tangent will not work
	    xInt = aPoint.getX();
	}
	else {
	    aSlope = PenroseTiling.tan(aRot);
	    aInt = aPoint.getY() - (aSlope * aPoint.getX()); // b = y - mx
	}
	
	if (bRot == 90)	{ // Horizontal
	    xInt = bPoint.getX();
	}
	else {
	    bSlope = PenroseTiling.tan(bRot);
	    bInt = bPoint.getY() - (bSlope * bPoint.getX());
	}
	
	if ( (aRot != 90) && (bRot != 90) ) {
	    // compute the x-intersection Point
	    xInt = ( (bInt - aInt) / (aSlope - bSlope) );
	}
	
	
	//System.out.println("aSlope: " + aSlope + "     aInt: " + aInt);
	//System.out.println("bSlope: " + bSlope + "     bInt: " + bInt);
	
	/*
	 * Now compute the y-intersection
	 * 
	 * From above:
	 * 	if aRot !=90, then aSlope and aInt are defined
	 * 	if bRot !=90, then bSlope and bInt are defined
	 * 	if either rotation is 90, then xInt is defined
	 * 	if neither rotation is 90, then xInt is defined
	 * 
	 * Therefore, we need only check the rotation, as xInt is ALWAYS defined
	 */
	
	if (aRot != 90)	{
	    // We cannot use a line if it has no slope
	    
	    // From above, if aRot !=90, then aSlope and aInt are defined
	    
	    // Just using good old y = mx + b form here
	    
	    yInt = aSlope * xInt + aInt;
	}
	else {
	    // at most one rotation can be 90, so we can assume that
	    // this will work if a's rotation is 90
	    // From above, if bRot !=90, then bSlope and bInt are defined
	    
	    yInt = bSlope * xInt + bInt;
	}
	
	return new Point2D.Double(xInt, yInt);
	
    }
    
    
    /**
     * <p>
     * Determines if a point is on a forced bar.
     * </p>
     *
     * @param p The point to check
     *
     * @return boolean True, if the point lies on a forced bar
     */
    public boolean isForced(Point2D p) {
	return isForced(p.getX(), p.getY());
    }
    

    /**
     * <p>
     * Determines if a point is on a forced bar.
     * </p>
     *
     * @param x The x coordinate to check
     * @param y The y coordinate to check
     *
     * @return boolean True, if the point lies on a forced bar
     */
    public boolean isForced(double x, double y)	{
	
	for (int i = 0; i < NUMSEQUENCES; i++) {
	    if (isForced(x, y, sequences[i])) {
		return true;
	    }
	}
	
	return false;
    }
    
    
    /**
     * <p>
     * Determines if a point is on a forced bar from the given sequence.
     * </p>
     *
     * @param p The point to check
     * @param ms The sequence to check
     *
     * @return boolean True, if the point lies on a forced bar from this
     * sequence
     */
    public boolean isForced(Point2D p, MusicalSequence ms) {
	return isForced(p.getX(), p.getY(), ms);
    }
    

    /**
     * <p>
     * Determines if a point is on a forced bar from the given sequence.
     * </p>
     *
     * @param x The x coordinate to check
     * @param y The y coordinate to check
     * @param ms The sequence to check
     *
     * @return boolean True, if the point lies on a forced bar from this
     * sequence
     */
    public boolean isForced(double x, double y, MusicalSequence ms) {

//System.err.println("Checking to see if a bar is forced on " + ms.rotation + ":");
//System.err.println("\tReal Point: (" + x + ", " + y + ")");
	
	Point2D forcedPoint = findNearestPoint(x, y, ms);
//System.err.println("\tTranslated to axis " + forcedPoint.toString());
	
	// Get the distance along the sequence
	double distance = getDistanceAlongSequence(forcedPoint, ms);
//System.err.println("\tDistance along sequence " + distance);
	
	// figure out what bar number that would be
	long barNum = ms.doubleToBar(distance);
//System.err.println("\tBarNum is " + barNum);
	
	// Alas, findBar will find the closest bar, but that might be far away.
	// We must confirm that the bar is actually close to our distance
	double barDist = ms.barToDouble(barNum);
	
	if (PenroseTiling.inRange(distance, barDist)) {
	    // And return whether it is forced or not
	    return ms.isForced(barNum);
	}
	else {
	    // not within tolerance
//System.out.println("Not within tolerance: " + distance + " and barDist " + barDist);
	    return false;
	}
	
    }
    
    
    /**
     * <p>
     * Takes the given point and sequence and returns the point after it has
     * been translated perpendicular to the axis of the sequence so that it
     * now lies on the axis itself.  This new translated point can now be
     * used to find the distance along the sequence that the original point
     * lies.
     * </p>
     *
     * @param p The point to translate
     * @param ms The sequence to translate against
     *
     * @return Point2D The point after it has been translated onto the axis
     */
    protected static Point2D findNearestPoint(Point2D p, MusicalSequence ms) {
	// Uses double method below

	return findNearestPoint(p.getX(), p.getY(), ms);
    }
    

    /**
     * <p>
     * Takes the given point and sequence and returns the point after it has
     * been translated perpendicular to the axis of the sequence so that it
     * now lies on the axis itself.  This new translated point can now be
     * used to find the distance along the sequence that the original point
     * lies.
     * </p>
     *
     * @param x The x coordinate to translate
     * @param y The y coordinate to translate
     * @param ms The sequence to translate against
     *
     * @return Point2D The point after it has been translated onto the axis
     */
    protected static Point2D findNearestPoint(double x, double y,
					      MusicalSequence ms) {
	
	/*
	 * Main idea:
	 * 	Construct a line through the Point2D that would be the bar in the sequence
	 * 	Construct a line to this line, along the axis of the sequence
	 * 	Return the intersection Point
	 */
	
	// We need two lines in "Point-rotation" form

	// First line: Bar in sequence through Point2D (x,y)
	Point2D barPoint = new Point2D.Double(x,y);

	// Bars are perpendicular to axis
	double barR = ms.rotation + 90;
	
	// Second line: Line from origin to the line we just constructed
	Point2D axisPoint = new Point2D.Double(ms.centerX, ms.centerY);
	double axisR = ms.rotation;
	
	// Now get their intersection:
	Point2D intersect = getIntersection(barPoint, barR,
					    axisPoint, axisR);
	
	return intersect;
	
    }
    
    
    /**
     * <p>
     * Returns true if these two points are the specified distance apart,
     * within some tolerance epislon.
     * </p>
     *
     * @param x1 The first x coordinate
     * @param y1 The first y coordinate
     * @param x2 The second x coordinate
     * @param y2 The second y coordinate
     * @param distance The distance to check
     * @param epsilon The tolerance value epsilon
     *
     * @return True, if the points are the specified distance apart,
     * within the tolerance given
     */
    protected boolean PointsWithinRange(double x1, double y1,
					double x2, double y2,
					double distance, double epsilon) {
	
	double actualDistance = Point2D.distance(x1, y1, x2, y2);

	return PenroseTiling.inRange(distance, actualDistance, epsilon);

    }


    /**
     * <p>
     * Returns an array of all the forced bars in the area specified.
     * The area to search is specified by two opposite corners of a
     * rectangle.  Any forced bars that intersect this rectangle are
     * returned.
     * </p>
     *
     * @param x1 The x coordinate of the first corner of the area
     * @param y1 The y coordinate of the first corner of the area
     * @param x2 The x coordinate of the second corner of the area
     * @param y2 The y coordinate of the second corner of the area
     * @param ms The sequence to get the bars from
     *
     * @return long[] An array of forced bar numbers, all of which
     * intersect the given area
     */
    public static long[] getForcedBars(double x1, double y1,
				       double x2, double y2,
				       MusicalSequence ms) {

	return getBars(x1, y1, x2, y2, ms, true);

    }


    /**
     * <p>
     * Returns an array of all the unforced bars in the area specified.
     * The area to search is specified by two opposite corners of a
     * rectangle.  Any unforced bars that intersect this rectangle are
     * returned.
     * </p>
     *
     * @param x1 The x coordinate of the first corner of the area
     * @param y1 The y coordinate of the first corner of the area
     * @param x2 The x coordinate of the second corner of the area
     * @param y2 The y coordinate of the second corner of the area
     * @param ms The sequence to get the bars from
     *
     * @return long[] An array of unforced bar numbers, all of which
     * intersect the given area
     */
    public static long[] getUnforcedBars(double x1, double y1,
				       double x2, double y2,
				       MusicalSequence ms) {

	return getBars(x1, y1, x2, y2, ms, false);

    }


    /**
     * <p>
     * Returns an array of all the bars (either forced or unforced)
     * in the area specified.
     * The area to search is specified by two opposite corners of a
     * rectangle.  Any bars that intersect this rectangle are
     * returned.  The boolean variable determines if forced or
     * unforced bars will be returned.
     * </p>
     *
     * @param x1 The x coordinate of the first corner of the area
     * @param y1 The y coordinate of the first corner of the area
     * @param x2 The x coordinate of the second corner of the area
     * @param y2 The y coordinate of the second corner of the area
     * @param ms The sequence to get the bars from
     * @param forced Whether to return forced or unforced bars
     *
     * @return long[] An array of unforced bar numbers, all of which
     * intersect the given area
     */
    protected static long[] getBars(double x1, double y1,
				    double x2, double y2,
				    MusicalSequence ms,
				    boolean forced) {

	// The largest distance along the sequence
	double largest = Double.MIN_VALUE;
	
	// The smallest distance along the sequence
	double smallest = Double.MAX_VALUE;

	double[] corners = new double[4];

	corners[0] = getDistanceAlongSequence(findNearestPoint(x1, y1, ms),ms);
	corners[1] = getDistanceAlongSequence(findNearestPoint(x2, y1, ms),ms);
	corners[2] = getDistanceAlongSequence(findNearestPoint(x1, y2, ms),ms);
	corners[3] = getDistanceAlongSequence(findNearestPoint(x2, y2, ms),ms);

	// Now get the largest and smallest distances along the sequence
	for (int i = 0; i < 4; i++) {
	    
	    if (corners[i] < smallest) {
		smallest = corners[i];
	    }
	    if (corners[i] > largest) {
		largest = corners[i];
	    }
	}
	    
	// Get the first and last bars:
	long firstBar = ms.doubleToBar(smallest);
	long lastBar = ms.doubleToBar(largest);

	if (forced) {
	    return ms.getForcedBars(firstBar, lastBar);
	}
	else {
	    return ms.getUnforcedBars(firstBar, lastBar);
	}

    }


    /**
     * <p>
     * Returns a collection of all the intersection points occuring inside
     * this area.  The area is defined as the points that mark the opposite
     * corners of a rectangle. The collection is sorted.
     * </p>
     *
     * @param x1 The x coordinate of the first corner of the area
     * @param y1 The y coordinate of the first corner of the area
     * @param x2 The x coordinate of the second corner of the area
     * @param y2 The y coordinate of the second corner of the area
     *
     * @return SortedSet A collection of all the intersection points
     */
    public SortedSet findIntersectionPoints(double x1, double y1,
					    double x2, double y2) {


	TreeSet points = new TreeSet();
	
	// Construct a bounding rectangle that we can use to see if
	// points fall within our area of consideration or not

	double lowerX, lowerY, rectWidth, rectHeight;

	if (x1 < x2) {
	    lowerX = x1;
	    rectWidth = x2 - x1;
	}
	else {
	    lowerX = x2;
	    rectWidth = x1 - x2;
	}

	if (y1 < y2) {
	    lowerY = y1;
	    rectHeight = y2 - y1;
	}
	else {
	    lowerY = y2;
	    rectHeight = y1 - y2;
	}

	Rectangle2D bounds = new Rectangle2D.Double(lowerX, lowerY,
						    rectWidth, rectHeight);

	if (PenroseTiling.DEBUG > 1) {
	    System.err.println("Checking from " + lowerX + ", " + lowerY +
			       " at a width of " + rectWidth +
			       " and height of " + rectHeight);
	    
	    System.err.println("Intercept rect is: " + bounds);
	}

	// Find all bars at a forced distance
	// We must do this for each set of bars, so lets keep several lists:
	
	long[][] bars = new long[NUMSEQUENCES][];
	
	for (int i = 0; i < NUMSEQUENCES; i++)	{
	    bars[i] = getForcedBars(x1, y1, x2, y2, sequences[i]);
	}

	// Now we have an array of sets of bars that are in the correct range
	// We need to check them all for intersections:
	
	for (int m = 0; m < (NUMSEQUENCES-1); m++) {

	    for (int n = m+1; n < NUMSEQUENCES; n++) {

		for (int a = 0; a < bars[m].length; a++) {

		    for (int b = 0; b < bars[n].length; b++) {

			// now get all the intersections of all the bars
			Point2D intersect = getIntersectionPoint(sequences[m],
								 bars[m][a],
								 sequences[n],
								 bars[n][b]);

			// check range
			if (bounds.contains(intersect.getX(),
					    intersect.getY())) {
			    // The point is in our area.

			    IntersectionPoint intersection =
				new IntersectionPoint(sequences[m],
						      bars[m][a],
						      sequences[n],
						      bars[n][b],
						      intersect);

			    intersection.addTo(points);
			}

		    }

		}

	    }

	}

	return points;
    }
    
    
    /**
     * <p>
     * Returns shapes that will draw all of the forced bars that fall inside
     * the specified area.  The area is defined as the points that mark the
     * opposite corners of a rectangle.
     * </p>
     *
     * @param x1 The x coordinate of the first corner of the area
     * @param y1 The y coordinate of the first corner of the area
     * @param x2 The x coordinate of the second corner of the area
     * @param y2 The y coordinate of the second corner of the area
     *
     * @return Line2D[] An array containing all the forced lines
     * that fall inside the specified area
     */
    public Line2D[] getForcedLines(double x1, double y1,
				   double x2, double y2) {

	return getLines(x1, y1, x2, y2, true);
    }


    /**
     * <p>
     * Returns shapes that will draw all of the unforced bars that fall inside
     * the specified area.  The area is defined as the points that mark the
     * opposite corners of a rectangle.
     * </p>
     *
     * @param x1 The x coordinate of the first corner of the area
     * @param y1 The y coordinate of the first corner of the area
     * @param x2 The x coordinate of the second corner of the area
     * @param y2 The y coordinate of the second corner of the area
     *
     * @return Line2D[] An array containing all the unforced lines
     * that fall inside the specified area
     */
    public Line2D[] getUnforcedLines(double x1, double y1,
				     double x2, double y2) {

	return getLines(x1, y1, x2, y2, false);
    }


    /**
     * <p>
     * Returns shapes that will draw all of the bars that fall inside
     * the specified area.  The area is defined as the points that mark the
     * opposite corners of a rectangle.  The forced variable determines
     * whether forced or unforced bars will be returned.
     * </p>
     *
     * @param x1 The x coordinate of the first corner of the area
     * @param y1 The y coordinate of the first corner of the area
     * @param x2 The x coordinate of the second corner of the area
     * @param y2 The y coordinate of the second corner of the area
     * @param forced Whether to return forced or unforced lines
     *
     * @return Line2D[] An array containing all the (un)forced lines
     * that fall inside the specified area
     */
    public Line2D[] getLines(double x1, double y1,
			      double x2, double y2,
			      boolean forced) {

	long[][] bars = new long[NUMSEQUENCES][];
	int total = 0;

	for (int i = 0; i < NUMSEQUENCES; i++) {
	    bars[i] = getBars(x1, y1, x2, y2, sequences[i], forced);
	    total += bars[i].length;
	}

	Line2D[] single = new Line2D[total];
	int count = 0;

	for (int j = 0; j < NUMSEQUENCES; j++) {

	    Line2D[] slice = sequences[j].getLines(bars[j], x1, y1, x2, y2);

	    for (int k = 0; k < slice.length; k++) {
		single[count] = slice[k];
		count++;
	    }
	}
	
	return single;
    }
    
    
    /**
     * <p>
     * Returns shapes that will draw all of the axes of the musical
     * sequences that intersect the given area.  The area is defined as 
     * the points that mark the opposite corners of a rectangle.
     * </p>
     *
     * @param x1 The x coordinate of the first corner of the area
     * @param y1 The y coordinate of the first corner of the area
     * @param x2 The x coordinate of the second corner of the area
     * @param y2 The y coordinate of the second corner of the area
     *
     * @return Line2D[] An array containing all the axes of the musical
     * sequences
     */
    public Line2D[] getAxes(double x1, double y1,
			    double x2, double y2) {
	
	Line2D[] axes = new Line2D[NUMSEQUENCES];

	for (int i = 0; i < axes.length; i++) {
	    axes[i] = sequences[i].getAxis(x1, y1, x2, y2);
	}

	return axes;
    }
    
    
    /**
     * <p>
     * Returns a string representation of this object.
     * </p>
     *
     * @return String The FiveFold as a string
     */
    public String toString() {
	
	return "FiveFold toString() Not implemented";
    }


    /**
     * <p>
     * Tests the class from the command line.
     * </p>
     *
     * @param args The command line arguments
     */
    public static void main(String args[]) {
	
	FiveFold penrose = new FiveFold();
	
	penrose.test();
	
	System.exit(0);
	
    }
    

    /**
     * <p>
     * Tests the class.
     * </p>
     */
    public void test() {
	// This should run some tests to check to see if the class is working
	
	System.out.println("\nTesting...");
	
	sequences[0].force(1,PenroseTiling.SHORTER);
	sequences[1].force(1,PenroseTiling.SHORTER);
	sequences[2].force(1,PenroseTiling.SHORTER);
	sequences[3].force(1,PenroseTiling.SHORTER);
	sequences[4].force(1,PenroseTiling.SHORTER);
	
	System.out.println("\nNow checking intersection between two sequences, first bar of each, both with short interval");
	
	for (int j = 0; j < 5; j++)	{
	    Point2D P = getIntersectionPoint(sequences[j], 1, sequences[(j+1)%5], 1);
	    System.out.println("IntersectionPoint between " + sequences[j].rotation + "," + sequences[(j+1)%5].rotation + " at " + P.toString());
	}
	
	System.out.println("\nNow finding all intersetctions 1.2 away from origin...");


	SortedSet intersections = findIntersectionPoints(-1.2, -1.2, 1.2, 1.2);
	java.util.Iterator setElts = intersections.iterator();
	
	while (setElts.hasNext())	{
	    IntersectionPoint I = (IntersectionPoint)(setElts.next());
	    System.out.println(I.toString());
	}

	System.out.println("\n");
    }
    
    
}



