// $Header: /opt/cvs/java/net/logn/penrose/IntersectionPoint.java,v 1.12 2001/02/08 20:08:55 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.penrose;

// AWT Geometry classes
import java.awt.geom.Point2D;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Rectangle2D;
// Collection interface used for storing points
import java.util.Collection;

/**
 * <p>
 * <b>IntersectionPoint</b> describes a point where two Ammann Bars
 * intersect.  The points have both five-dimensional (sequence) and
 * two-dimensional (real) coordinates.  The points also know how to
 * sift themselves into "boxes"; essentially grouping themselves with
 * other nearby points.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.12 $
 *
 * Last Modified $Date: 2001/02/08 20:08:55 $ by $Author: jhealy $
 */
public class IntersectionPoint extends Point2D.Double implements Comparable {

    // An intersection requires two AmmannBars
    /** Sequence in this intersection with the lowest angle of rotation */
    protected MusicalSequence seq1;

    /** Bar on the sequence with the lowest angle of rotation */
    protected long bar1;

    /** Sequence in this intersection with the greatest angle of rotation */
    protected MusicalSequence seq2;

    /** Bar on the sequence with the greatest angle of rotation */
    protected long bar2;

    /** Box layer (distance from origin of coordinate system) */
    protected double boxLayer;

    /** Box theta (rotation about the origin of coordinate system) */
    protected double boxTheta;

    // Because a point may need to have extra copies of itself for overlapping
    // boxes, we must be able to store points that overlap with boxes to the
    // right or bottom of this box.
    private IntersectionPoint duplicateRight = null;
    private IntersectionPoint duplicateBottom = null;
    private IntersectionPoint duplicateDiagonal = null;


    /**
     * <p>
     * Constructs a point that is not complete or valid.  Should only be
     * used to hold impartial matches.
     * </p>
     *
     * @param point The 2-D point wrapped up in this object
     */
    public IntersectionPoint(Point2D point) {

	this.x = point.getX();
	this.y = point.getY();

	seq1 = null;
	bar1 = 0;
	seq2 = null;
	bar2 = 0;

	boxLayer = -1.0;
	boxTheta = -1.0;

    }


    /**
     * <p>
     * Constructs a point, given two Ammann Bars and where they intersect
     * </p>
     *
     * @param firstSeq The first sequence in the intersection
     * @param firstBar The first bar in the intersection
     * @param secondSeq The second sequence in the intersection
     * @param secondBar The second bar in the intersection
     * @param point The point of real intersection
     */
    public IntersectionPoint(MusicalSequence firstSeq, long firstBar,
			     MusicalSequence secondSeq, long secondBar,
			     Point2D point) {

	this(firstSeq, firstBar, secondSeq, secondBar,
	     point.getX(), point.getY());
    }


    /**
     * <p>
     * Constructs a point, given two Ammann Bars and where they intersect
     * </p>
     *
     * @param firstSeq The first sequence in the intersection
     * @param firstBar The first bar in the intersection
     * @param secondSeq The second sequence in the intersection
     * @param secondBar The second bar in the intersection
     * @param xVal The x value of the intersection point
     * @param yVal The y value of the intersection point
     */
    public IntersectionPoint(MusicalSequence firstSeq, long firstBar,
			     MusicalSequence secondSeq, long secondBar,
			     double xVal, double yVal) {

	// Make sure that the lowest-numbered sequence goes first

	if (firstSeq.rotation < secondSeq.rotation) {
	    seq1 = firstSeq;
	    seq2 = secondSeq;
	    bar1 = firstBar;
	    bar2 = secondBar;
	}
	else {
	    seq2 = firstSeq;
	    seq1 = secondSeq;
	    bar2 = firstBar;
	    bar1 = secondBar;
	}

	// real coordinates:
	x = xVal;
	y = yVal;

	// now compute the box that this point belongs to
	computeBoxValues();
    }


    /**
     * <p>
     * Constructs an Intersection from another, but using a different
     * virtual box.  This is used when we need to make a copy of a point
     * for an overlapping virtual box.
     * </p>
     *
     * @param point The point to base this point off of
     * @param layer The new box layer for this point
     * @param theta The new box theta for this point
     */
    private IntersectionPoint(IntersectionPoint point,
			      double layer, double theta) {
	
	this.seq1 = point.seq1;
	this.seq2 = point.seq2;
	this.bar1 = point.bar1;
	this.bar2 = point.bar2;
	this.x = point.x;
	this.y = point.y;

	this.boxLayer = layer;
	this.boxTheta = theta;

    }


    /**
     * <p>
     * Constructs this Intersection Point's "box values".  If we were
     * to divide the plane up into equal-sized boxes, then each point
     * would fall into a box.  This method computes the box that the
     * point would fall into and stores this unique box information.
     * </p>
     *
     * <p>
     * When the points are sorted, they will sort according to their
     * box values, thus lumping together all point that are near each
     * other in the plane.
     * </p>
     */
    protected void computeBoxValues() {

	// First, get how many boxes between the origin and the point
	// truncate to nearest integer
	double xBoxes = (x - PenroseTiling.BOX_ORIGIN) / PenroseTiling.BOX_DIM;
	double yBoxes = (y - PenroseTiling.BOX_ORIGIN) / PenroseTiling.BOX_DIM;

	// Now, get the integer component

	double xFloor = Math.floor(xBoxes);
	double yFloor = Math.floor(yBoxes);

	// And the remainder
	double xRemainder = xBoxes - xFloor;
	double yRemainder = yBoxes - yFloor;

	// If the remainder is greater than (BOX_DIM - OVERLAP), then 
	double xOverflow = (xRemainder * PenroseTiling.BOX_DIM) + PenroseTiling.BOX_OVERLAP;
	double yOverflow = (yRemainder * PenroseTiling.BOX_DIM) + PenroseTiling.BOX_OVERLAP;


	
	if ( (xOverflow > PenroseTiling.BOX_DIM) &&
	     (yOverflow > PenroseTiling.BOX_DIM) ) {

	    // we're into the next block
	    if (PenroseTiling.DEBUG > 5) {
		System.err.println("We've overflowed in both X and Y; duplicating point");
	    }

	    double[] newBox = getBoxInfo((xFloor + 1.0), (yFloor + 1.0));

	    this.duplicateDiagonal = new IntersectionPoint(this, 
							   newBox[0], newBox[1]);

	}
	if (xOverflow > PenroseTiling.BOX_DIM) {
	    // we're into the next block
	    if (PenroseTiling.DEBUG > 5) {
		System.err.println("We've overflowed in X; duplicating point");
	    }
	    
	    double[] newBox = getBoxInfo((xFloor + 1.0), yFloor);
	    
	    this.duplicateRight = new IntersectionPoint(this, 
							newBox[0], newBox[1]);
	    
	}
	if (yOverflow > PenroseTiling.BOX_DIM) {
	    // we're into the next block
	    if (PenroseTiling.DEBUG > 5) {
		System.err.println("We've overflowed in Y; duplicating point");
	    }
	    
	    double[] newBox = getBoxInfo(xFloor, (yFloor + 1.0));
	    
	    this.duplicateBottom = new IntersectionPoint(this, 
							 newBox[0], newBox[1]);
	    
	}
	
	
	if (PenroseTiling.DEBUG > 3) {
	    System.err.println("Boxes between origin and point: " + xBoxes +
			       ", " + yBoxes);
	    
	    System.err.println("Overflow: " + xOverflow + ", " + yOverflow);
	    
	    System.err.println("Actual Boxes: " + xFloor + ", " + yFloor);
	}

	double[] boxInfo = getBoxInfo(xFloor, yFloor);

	boxLayer = boxInfo[0];
	boxTheta = boxInfo[1];
    }


    /**
     * <p>
     * Returns a rectangle that shows this points virtual box.
     * </p>
     */
    public Rectangle2D getBoxRect() {

	// First, get how many boxes between the origin and the point
	// truncate to nearest integer
	double xBoxes = (x - PenroseTiling.BOX_ORIGIN) / PenroseTiling.BOX_DIM;
	double yBoxes = (y - PenroseTiling.BOX_ORIGIN) / PenroseTiling.BOX_DIM;

	// Now, get the integer component

	double xFloor = Math.floor(xBoxes);
	double yFloor = Math.floor(yBoxes);

	// And the remainder
	double xRemainder = xBoxes - xFloor;
	double yRemainder = yBoxes - yFloor;

	// If the remainder is greater than (BOX_DIM - OVERLAP), then 
	double xOverflow = (xRemainder * PenroseTiling.BOX_DIM) + PenroseTiling.BOX_OVERLAP;
	double yOverflow = (yRemainder * PenroseTiling.BOX_DIM) + PenroseTiling.BOX_OVERLAP;

	// don't return a rect if we're not sure which box this point goes in

	if (xOverflow > PenroseTiling.BOX_DIM) {
	    return null;
	}
	
	if (yOverflow > PenroseTiling.BOX_DIM) {
	    return null;
	}

	// otherwise, send the rect back
	
	return new Rectangle2D.Double(((xFloor * PenroseTiling.BOX_DIM) +
				       PenroseTiling.BOX_ORIGIN -
				       PenroseTiling.BOX_OVERLAP),
				      
				      ((yFloor * PenroseTiling.BOX_DIM) +
				       PenroseTiling.BOX_ORIGIN -
				       PenroseTiling.BOX_OVERLAP),

				      (PenroseTiling.BOX_DIM +
				       PenroseTiling.BOX_OVERLAP),
				      
				      (PenroseTiling.BOX_DIM +
				       PenroseTiling.BOX_OVERLAP));
    }


    /**
     * <p>
     * Constructs this Intersection Point's "box values".  If we were
     * to divide the plane up into equal-sized boxes, then each point
     * would fall into a box.  This method computes the box that the
     * point would fall into and stores this unique box information.
     * </p>
     * 
     * @param xBox The number of boxes in the x direction
     * @param yBox The number of boxes in the y direction
     *
     * @return double[] An array with the box layer in position 0
     * and the box theta in position 1
     */
    protected static double[] getBoxInfo(double xBox, double yBox) {

	double[] info = new double[2];

	info[0] = Math.max( Math.abs(xBox), Math.abs(yBox) );

	// now compute the theta of the box
	info[1] = PenroseTiling.atan2(xBox, yBox);

	// make sure the value is between 0 and 360
	if (info[1] < 0.0) {
	    info[1] += 360;
	}

	if (PenroseTiling.DEBUG > 3) {
	    System.err.println("I think that coordinates " + xBox + ", " + yBox +
			       " form an angle of " + info[1] + " degrees");
	}

	return info;

    }


    /**
     * <p>
     * Compares this point with the specified object for order. 
     * Returns a negative integer, zero, or a positive integer as this 
     * object is less than, equal to, or greater than the specified object.
     * </p>
     *
     * <p>
     * Order, in this case, is defined by the box layer.  If the layers are
     * the same, then the box theta is used.  If those are equal, then the
     * ordering of points is based upon the musical sequence rotations, and
     * then bar numbers.  The lowest sequence rotation(s) are used first, and
     * then the bar numbers are used.  If all those are equal, then the
     * points are the same.
     * </p>
     *
     * @param o The object to compare to
     *
     * @return int 0 for equals, < 0 for less than, and > 0 for greater than
     */
    public int compareTo(Object o) {

	IntersectionPoint that = (IntersectionPoint)o;

	// check layers
	
	if (this.boxLayer < that.boxLayer) {
	    return -1;
	}
	else if (that.boxLayer < this.boxLayer) {
	    return 1;
	}
	else {
	    // check thetas
	    
	    if (this.boxTheta < that.boxTheta) {
		return -1;
	    }
	    else if (that.boxTheta < this.boxTheta) {
		return 1;
	    }
	    else {
		// check the primary sequences
		
		if (this.seq1.rotation < that.seq1.rotation) {
		    return -1;
		}
		else if (that.seq1.rotation < this.seq1.rotation) {
		    return 1;
		}
		else {
		    // check the secondary sequences
		    if (this.seq2.rotation < that.seq2.rotation) {
			return -1;
		    }
		    else if (that.seq2.rotation < this.seq2.rotation) {
			return 1;
		    }
		    else {
			// check the primary bars
			
			if (this.bar1 < that.bar1) {
			    return -1;
			}
			else if (that.bar1 < this.bar1) {
			    return 1;
			}
			else {
			    // check the secondary bars
			    if (this.bar2 < that.bar2) {
				return -1;
			    }
			    else if (this.bar2 > that.bar2) {
				return 1;
			    }
			    else {
				// EQUAL
				return 0;
			    } // end if secondary bars are the same
			} // end if primary bars are the same

		    } // end if secondary sequences are the same
		} // end if primary sequences are the same

	    } // end if boxThetas are the same
	} // end if boxLayers are the same
	
    }
    

    /**
     * <p>
     * Returns true if this point equals another point.  This does <b>not</b>
     * agree with compareTo() on the issue of equality.  compareTo() insists
     * that two points must have the same bar and sequence intersections, and
     * must also be in the same virtual box.  equals(), however, only requires
     * that the sequence addresses agree; the box is not relevant.
     * </p>
     *
     * @param o The object to compare to
     *
     * @return boolean True, if the objects are equal
     */
    public boolean equals(Object o) {

	IntersectionPoint that = (IntersectionPoint)o;

	if (this.seq1.rotation != that.seq1.rotation) {
	    return false;
	}
	if (this.seq2.rotation != that.seq2.rotation) {
	    return false;
	}
	if (this.bar1 != that.bar1) {
	    return false;
	}
	if (this.bar2 != that.bar2) {
	    return false;
	}
	// we made it!
	return true;
    }
    

    /**
     * <p>
     * Because our virtual boxes may overlap, a point may exist simultaneously
     * inside two boxes.  To support this, a point must be allowed to add
     * more than one copy of itself to the store, each with different box
     * values.  To do so, the point supports "adding itself" (and possibly
     * other copies of itself) to the store.
     * </p>
     *
     * <p>
     * This method adds an intersection point, along with any duplicates
     * that are necessary, to the store.
     * </p>
     *
     * @param store A data structure supporting the Collection interface
     *
     * @return boolean The return value from adding this point to the store
     */
    public boolean addTo(Collection store) {

	boolean success = false;

	success = store.add(this);

	if (duplicateRight != null) {
	    success = success && store.add(duplicateRight);
	}

	if (duplicateBottom != null) {
	    success = success && store.add(duplicateBottom);
	}

	if (duplicateDiagonal != null) {
	    success = success && store.add(duplicateDiagonal);
	}

	return success;
    }


    /**
     * <p>
     * Returns a shape that identifies this point on the plane.
     * </p>
     *
     * @return Ellipse2D The representation of this point on the screen
     */
    public Ellipse2D getShape() {

	return new Ellipse2D.Double((x - 0.1), (y - 0.1), 0.2, 0.2);
    }


    /**
     * <p>
     * Tests the class on the command line
     * </p>
     *
     * @param args Command line arguments
     */
    public static void main(String[] args) {

	System.out.println("\nRunning automatic tests\n\n");
	
	java.util.Vector points = new java.util.Vector(25);
	
	MusicalSequence primary = new MusicalSequence(0.0, 0.0, 0.0);
	MusicalSequence secondary = new MusicalSequence(0.0, 0.0, 72.0);
	
	primary.force(10, PenroseTiling.LONGER);
	secondary.force(10, PenroseTiling.SHORTER);
	
	for (int i = 0; i < 100; i+=10) {
	    for (int j = 0; j < 100; j+=10) {
		
		IntersectionPoint point = new IntersectionPoint(primary, i, secondary, j, i, j);
		if (!point.addTo(points)) {
		    System.out.println("Error Adding in addTo()!!!");
		}
	    }
	}
	
	// now sort the points
	java.util.Collections.sort(points);
	
	// and print them out
	for (int k = 0; k < points.size(); k++) {
	    IntersectionPoint p = (IntersectionPoint)points.elementAt(k);
	    
	    System.out.println("Point Box Layer: " + p.boxLayer +
			       "  Theta: " + p.boxTheta);
	    
	}
	
    }

}
