// $Header: /opt/cvs/java/net/logn/penrose/MusicalSequence.java,v 1.7 2001/02/17 20:43:59 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.penrose;

// Used to mark where the sequence originates
import java.awt.geom.Point2D;
import java.awt.geom.Line2D;
// Allows us to save the object out to a file
import java.io.Serializable;

/**
 * <p>
 * A <b>Musical Sequence</b> is a set of Ammann Bars that are perpendicular
 * to a single axis.  This axis stretches infinitely in both directions in
 * the plane, and has an angle of 0, 72, 144, 216, or 288 degrees.
 * </p>
 *
 * <p>
 * Additionally, a Musical Sequence has the notion of a "center" in the plane.
 * We must be able to describe where a Musical Sequence originates, and so
 * these two-dimensional coordinates allow us to specify the center.
 * </p>
 *
 * <p>
 * To compute the Musical Sequence values, we use Minnick's method (please
 * see Minnick 1998).  Briefly stated, this is how her method works.  We
 * store the sequence as two lines, each with slope Tau, with y-intercepts
 * between 0 and 1.  To determine if a bar N is forced, we consult the
 * x-coodinate N and examine the y-value at that point.  If the y-values
 * from both lines round to the same point, then the bar is forced.
 * </p>
 *
 * <p>
 * We store the lines here as integer values, using a point-intercept form
 * (since the slope is always known).  This eliminates as much rounding
 * error as possible.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.7 $
 *
 * Last Modified $Date: 2001/02/17 20:43:59 $ by $Author: jhealy $
 */
public class MusicalSequence implements Serializable {
    
    //	We represent the lines an intercept form, intercepting some 
    //	coordinate (X,Y) given slope Tau.
    
    /** The "upper" line's X coordinate */
    protected long upperX;

    /** The "upper" line's Y coordinate */
    protected long upperY;

    /** The "lower" line's X coordinate */
    protected long lowerX;

    /** The "lower" line's Y coordinate */
    protected long lowerY;
    
    /** The x coordinate of the center of the Musical Sequence */
    protected double centerX; // 2D origin of the sequence

    /** The y coordinate of the center of the Musical Sequence */
    protected double centerY;

    /** The rotation of the Musical Sequence */
    protected double rotation;
    

    /**
     * <p>
     * Constructor.  Places sequence at (0, 0) with no rotation and no
     * forced bars.
     * </p>
     */
    public MusicalSequence() {
	
	upperX = 0;
	upperY = 1;
	lowerX = 0;
	lowerY = 0;
	
	centerX = 0.0;
	centerY = 0.0;
	rotation = 0.0;
    }


    /**
     * <p>
     * Constructor.  Creates an unforced musical sequence at the coordinates
     * specified and with the rotation specified.
     * </p>
     *
     * @param x The x coordinate of the center of the sequence
     * @param y The y coordinate of the center of the sequence
     * @param r The rotation of the sequence
     */
    public MusicalSequence(double x, double y, double r) {

	this();

	centerX = x;
	centerY = y;
	rotation = r;
    }
    

    /**
     * <p>
     * Sets the location of the zeroth bar to be the distance specified
     * from the origin.
     * </p>
     *
     * @param distance The distance along the sequence to force the bar
     */
    public void setZeroBar(double distance) {
	
	centerX = distance * PenroseTiling.cos(rotation);
	centerY = distance * PenroseTiling.sin(rotation);

    }
    
    
    /**
     * <p>
     * Returns the center X coordinate.
     * </p>
     *
     * @return double The center X coordinate
     */
    public double getCenterX() {
	
	return centerX;	
    }
    
    
    /**
     * <p>
     * Returns the center Y coordinate.
     * </p>
     *
     * @return double The center Y coordinate
     */
    public double getCenterY() {
	
	return centerY;
    }
    
    
    /**
     * <p>
     * Returns the center of the sequence as a point.
     * </p>
     *
     * @return Point2D The center point of the sequence
     */
    public Point2D getCenter() {
	
	return new Point2D.Double(centerX, centerY);
    }
    
    
    /**
     * <p>
     * Returns the rotation of this sequence.
     * </p>
     *
     * @return double The rotation of this sequence
     */
    public double getRotation () {
	
	return rotation;
    }
    

    /**
     * <p>
     * Returns the X coordinate of the "upper" line.
     * </p>
     *
     * @return long The X coordinate of the "upper" line
     */
    public long getUpperX() {
	
	return upperX;
    }
    

    /**
     * <p>
     * Returns the Y coordinate of the "upper" line.
     * </p>
     *
     * @return long The Y coordinate of the "upper" line
     */
    public long getUpperY() {
	
	return upperY;
    }
    

    /**
     * <p>
     * Returns the X coordinate of the "lower" line.
     * </p>
     *
     * @return long The X coordinate of the "lower" line
     */
    public long getLowerX() {
	
	return lowerX;
    }
    

    /**
     * <p>
     * Returns the Y coordinate of the "lower" line.
     * </p>
     *
     * @return long The Y coordinate of the "lower" line
     */
    public long getLowerY() {
	
	return lowerY;
    }
    

    /**
     * <p>
     * Rounds a value down to the next lowest integer value
     * "Closed" refers to the fact that it will also round
     * down a value that is of integer value (that is,
     * truncate(2.0) returns 1.0), thus keeping the interval closed.
     * </p>
     *
     * @param value The value to truncate
     *
     * @return long The truncated value
     */
    protected static long truncateClosed(double value) {
	
	long temp;
	
	temp = (long)(Math.floor(value));
	
	if (temp == value) {
	    return (temp - 1);
	}
	else {
	    return temp;
	}
	
    }
    

    /**
     * <p>
     * Rounds a value down (truncates) to the next lowest 
     * integer value.  Does not change the value if it is already
     * an integer.  Thus, the interval remains "open".
     * </p>
     *
     * @param value The value to truncate
     *
     * @return long The truncated value
     */
    protected static long truncateOpen(double value) {
	
	return (long)(Math.floor(value));
	
    }
    

    /**
     * <p>
     * Returns the y-intercept of the "upper" bounding line.
     * </p>
     *
     * @return double The y-intercept of the "upper" bounding line
     */
    protected double findUpperIntercept() {

	return (upperY - (upperX * PenroseTiling.TAU));
    }
    

    /**
     * <p>
     * Returns the y-intercept of the "lower" bounding line.
     * </p>
     *
     * @return double The y-intercept of the "lower" bounding line
     */
    protected double findLowerIntercept() {
	
	return (lowerY - (lowerX * PenroseTiling.TAU));
    }
    
    
    public double Interval(String dont, String need, String anymore) {
	// Pre: none
	// Post: Returns the interval between the two intercepts

	return (findUpperIntercept() - findLowerIntercept());
    }


    /**
     * <p>
     * Returns the point in our 2-D lattice where our
     * c-line passes to intercept this x value.  <longer> determines
     * if we should use the Upper or Lower bound to compute this value.
     * </p>
     *
     * @param x The x value to intercept
     * @param longer Whether to make the interval longer or not
     *
     * @return double The y-value of the intercerpt point
     */
    protected double findPoint(long x, boolean longer) {
	
	double intercept;
	
	if (longer == PenroseTiling.LONGER)	{
	    intercept = findUpperIntercept();
	}
	else	{
	    intercept = findLowerIntercept();
	}
	
	return ((x * PenroseTiling.TAU) + intercept);
    }
    

    /**
     * <p>
     * Returns the Upper Y value at an arbitrary bar number.
     * </p>
     *
     * @param barNum The bar number to get the y value for
     *
     * @return long The upper y value at barNum
     */
    protected long findUpperPoint(long barNum) {
	// Find the Point, and truncate it to an integer
	// Use a closed interval for upper Points (truncateClosed)
	
	return truncateClosed(findPoint(barNum, PenroseTiling.LONGER));
    }
    
    
    /**
     * <p>
     * Returns the Lower Y value at an arbitrary bar number.
     * </p>
     *
     * @param barNum The bar number to get the y value for
     *
     * @return long The lower y value at barNum
     */
    protected long findLowerPoint(long barNum) {
	// Find the Point, and truncate it to an integer
	// Use an open interval for lower Points (truncateOpen)
	
	return truncateOpen(findPoint(barNum, PenroseTiling.SHORTER));
    }
    
    
    /**
     * <p>
     * Forces a bar to be a certain interval away from the preceeding
     * bar.  The length of the interval is made to match the one specified.
     * </p>
     *
     * @param barNum The bar number to force
     * @param makeLong Whether to make the interval long or not
     */
    public void force(long barNum, boolean makeLong) {
	
	// First, we have to find what Y values we might have:

	long longer = findUpperPoint(barNum);
	long shorter = findLowerPoint(barNum);
	
	// The bar must not already be forced
	if (longer != shorter)	{
	    
	    // note that the cases are reveresed for negative bars
	    if ( ((makeLong == PenroseTiling.LONGER) && (barNum >= 0)) ||
		 ((makeLong == PenroseTiling.SHORTER) && (barNum < 0)) ) {

		// force the bar to be farther away, by raising the lower bound
		
		lowerX = barNum;
		lowerY = longer;
		
	    }
	    else {
		// force the bar to be shorter, by lowering the upper bound
		
		upperX = barNum;
		upperY = longer;
		
	    }
	}
    }
    

    /**
     * <p>
     * Determines if a specific bar is forced.
     * </p>
     *
     * @param barNum The bar number to check
     *
     * @return boolean True, if the bar is forced
     */
    public boolean isForced (long barNum) {
	
	long longer = findUpperPoint(barNum);
        long shorter = findLowerPoint(barNum);
	
	return (longer == shorter);
    }
    

    /**
     * <p>
     * Returns the distance from center to specified bar
     * abstractly represented as the number of long plus the
     * number of short bars that would fit in the distance.
     * The specified bar <b>is assumed to be forced</b>.
     * </p>
     *
     * @param barNum The bar number to check
     *
     * @return String The distance, in the form "nL + mS"
     */
    public String distanceTo (long barNum) {
	
	//	given the Nth bar, with Y integer value M, the number of longs
	//	are given by M-N and the number of shorts are given by 2N-M
	// (Many thanks to prof. Bailey for that little formula...)
	
	//	Get the Y value of this bar:
	//	(can assume LONGER because the bar is forced, and is therefore
	//	the same as assuming SHORTER)
	
	long yVal = findUpperPoint(barNum);
	
	long shortBars = ((2 * barNum) - yVal);
	long longBars = (yVal - barNum);
	
	return (shortBars + "s + " + longBars + "l");
    }
    

    public double offsetDistance(String deprecated) {
	// Pre: none
	// Post: Returns the straight-line distance from the center of
	//	this musical sequence to the origin on the plane
	
	return 0.0;
	//	return PenroseTiling.distance(0, 0, centerX, centerY);
    }
    
    
    /**
     * <p>
     * Returns a 2D distance from the origin of the sequence to where this
     * bar intersects the axis.
     * </p>
     *
     * @param barNum The bar number to compute the distance to
     *
     * @return double The distance to the intersection of the bar and the axis
     */
    public double barToDouble(long barNum) {
	
	long yVal = findUpperPoint(barNum);
	long shortBars = ((2 * barNum) - yVal);
	long longBars = (yVal - barNum);

	return ((PenroseTiling.SCALE * shortBars) +
		(PenroseTiling.SCALE * PenroseTiling.TAU * longBars));
	
    }
    
    
    /**
     * <p>
     * Finds the bar number that is closest to this distance from the center
     * of the sequence.
     * </p>
     *
     * @param barNum The bar number to check
     *
     * @return long The bar that is closest to this distance from the center
     */
    public long doubleToBar(double distance) {
	
	/*
	 * Here is the "work" to solve this problem:
	 *
	 * Short distance = 1 * SCALE
	 * Long distance = T[AU] * SCALE
	 * 
	 * distance = x(1) + y(T)
	 * y/x = T/1
	 * y = xT
	 * 
	 * distance = x + xTT  (TT = T+1)
	 * 
	 * distance = x(1 + T + 1)
	 * distance = x(T + 2)
	 * 
	 * x = distance/(T + 2)
	 * y = x*T
	 * 
	 */
	
	double numShort = (distance / (PenroseTiling.SHORT +
				       (PenroseTiling.TAU * PenroseTiling.LONG)) );
	double numLong = (numShort * PenroseTiling.TAU);

	
	return ((long)Math.round(numShort + numLong));
	
    }
    
    
    /**
     * <p>
     * Returns the difference (absolute value) of the distance
     * provided and the distance computed by using the number of short 
     * and long bars given.
     * </p>
     *
     * @param S The number of short bars
     * @param L The number of long bars
     * @param d The distance expressed in real coordinates
     *
     * @return double The difference between the actual distance and the
     * distance defined in long and short bars
     */
    private double ddist(long S, long L, double d) {

	double mydist = (S + (PenroseTiling.TAU * L));

	double temp = Math.abs(d - mydist);

	return temp;
    }


    /**
     * <p>
     * Forces a bar to be forced at a given distance from
     * the zeroth bar of the sequence.  Returns true only
     * if bar wasn't already forced.
     * </p>
     *
     * @param barNum The bar number to check
     *
     * @return boolean True, if the bar wasn't already forced
     */
    public boolean forceAtDistance (double distance) {

	// Distance to a bar equals NumShort + NumLong*TAU
	// and numshort/numlong = 1/tau

	double scaled = distance / PenroseTiling.SCALE;

	double numShort = (scaled / (2 + PenroseTiling.TAU));
	double numLong = (numShort * PenroseTiling.TAU);

	// Since these values are not integers, we have to convert them
	// to approximate integer values and then test them:

	long[] shorts = new long[2];
	long[] longs = new long[2];

	shorts[0] = (long)(Math.floor(numShort));
	shorts[1] = shorts[0] + 1;
	longs[0] = (long)(Math.floor(numLong));
	longs[1] = longs[0] + 1;

//	System.out.println(shorts[0]+","+shorts[1]+" "+longs[0]+","+longs[1]);
	// Now there are four different values that we can compute.
	// We'll find the one that is the closest match for our distance,
	// and use it.

	int bestS = 0;
	int bestL = 0;

	for (int i = 0; i < 2; i++) {
	    for (int j = 0; j < 2; j++) {
		if (ddist(shorts[i],longs[j],scaled) <= ddist(shorts[bestS],longs[bestL],scaled)) {
//		    System.out.println("ddist="+ddist(shorts[i],longs[j],scaled));
		    bestS = i;
		    bestL = j;
		}
	    }
	}

//System.err.print("Best is " + bestS + "," + bestL + " (" + ddist(shorts[bestS],longs[bestL],scaled) + ") ");
	// So now we have the best possible integer lattice point.  We just
	// have to force it now.

	long bars = shorts[bestS] + longs[bestL];
	boolean alreadyForced = false;

	if ((bestL == 0) && (bars > -1)) { // if we rounded down, in the positive direction
	    alreadyForced = isForced(bars);
	    force(bars,PenroseTiling.SHORTER);
//System.err.println("Forced positive short");
	}
	else if ((bestL == 1) && (bars > -1)) { // if we rounded up, in the positive direction
	    alreadyForced = isForced(bars);
	    force(bars,PenroseTiling.LONGER);
//System.err.println("Forced positive long");
	}
	else if ((bestL == 0) && (bars < 0)) { // if we rounded down, in the negative direction
	    alreadyForced = isForced(bars);
	    force(bars,PenroseTiling.LONGER);
//System.err.println("Forced negative long");
	}
	else { // we rounded up, in the negative direction
	    alreadyForced = isForced(bars);
	    force(bars,PenroseTiling.SHORTER);
//System.err.println("Forced negative short");
	}

	return !alreadyForced;
    }
    

    /**
     * <p>
     * Returns the bar numbers of all the unforced bars between first and
     * last (inclusive).
     * </p>
     *
     * @param first The first bar to check
     * @param last The last bar to check
     *
     * @return long[] An array of bar numbers, all of which are unforced
     */
    public long[] getUnforcedBars(long first, long last) {

	long[] temp = new long[(int)((last - first) + 1)];

	int count = 0;

	for (long i = first; i <= last; i++) {
	    
	    if (!isForced(i)) {
		temp[count] = i;
		count++;
	    }
	}

	long[] bars = new long[count];
	
	for (int j = 0; j < count; j++) {
	    bars[j] = temp[j];
	}

	return bars;
    }
    

    /**
     * <p>
     * Returns shapes to draw all the unforced lines between first and
     * last (inclusive).
     * </p>
     *
     * @param first The first bar to check
     * @param last The last bar to check
     *
     * @return Line2D[] Array of shapes to draw all unforced lines
     */
    public Line2D[] getUnforcedLines(long first, long last) {

	long[] bars = getUnforcedBars(first, last);

	return getLines(bars);
    }
    

    /**
     * <p>
     * Returns the bar numbers of all the forced bars between first and
     * last (inclusive).
     * </p>
     *
     * @param first The first bar to check
     * @param last The last bar to check
     *
     * @return long[] An array of bar numbers, all of which are forced
     */
    public long[] getForcedBars(long first, long last) {

	long[] temp = new long[(int)((last - first) + 1)];

	int count = 0;

	for (long i = first; i <= last; i++) {
	    
	    if (isForced(i)) {
		temp[count] = i;
		count++;
	    }
	}

	long[] bars = new long[count];
	
	for (int j = 0; j < count; j++) {
	    bars[j] = temp[j];
	}

	return bars;
    }
    

    /**
     * <p>
     * Returns shapes to draw all the forced lines between first and
     * last (inclusive).
     * </p>
     *
     * @param first The first bar to check
     * @param last The last bar to check
     *
     * @return Line2D[] Array of shapes to draw all forced lines
     */
    public Line2D[] getForcedLines(long first, long last) {

	long[] bars = getForcedBars(first, last);

	return getLines(bars);
    }
    

    /**
     * <p>
     * Converts the bars into a lines suitable for drawing.
     * </p>
     *
     * @param bars The bars to render
     *
     * @return Line2D[] Array of Shapes to draw the bars with
     */
    public Line2D[] getLines(long[] bars) {

	Line2D[] lines = new Line2D[bars.length];

	for (int i = 0; i < bars.length; i++) {

	    lines[i] = getLine(bars[i]);

	}

	return lines;
    }
    

    /**
     * <p>
     * Converts the bars into lines suitable for drawing.  Clips the
     * lines to the bounding box provided.
     * </p>
     *
     * @param bars The bars to render
     * @param x1 The x of the first corner of the bounding box
     * @param y1 The y of the first corner of the bounding box
     * @param x2 The x of the second corner of the bounding box
     * @param y2 The y of the second corner of the bounding box
     *
     * @return Line2D[] Shape to draw the bar with
     */
    public Line2D[] getLines(long[] bars,
			     double x1, double y1,
			     double x2, double y2) {
	
	Line2D[] lines = new Line2D[bars.length];
	
	for (int i = 0; i < bars.length; i++) {

	    lines[i] = getLine(bars[i], x1, y1, x2, y2);

	}

	return lines;
    }
    

    /**
     * <p>
     * Converts the bar into a line suitable for drawing.
     * </p>
     *
     * @param bar The bar to render
     *
     * @return Line2D Shape to draw the bar with
     */
    public Line2D getLine(long bar) {

	return getLine(bar, -10000, -10000, 10000, 10000);
    }
    

    /**
     * <p>
     * Converts the bar into a line suitable for drawing.  Clips the
     * line to the bounding box provided.
     * </p>
     *
     * @param bar The bar to render
     * @param x1 The x of the first corner of the bounding box
     * @param y1 The y of the first corner of the bounding box
     * @param x2 The x of the second corner of the bounding box
     * @param y2 The y of the second corner of the bounding box
     *
     * @return Line2D Shape to draw the bar with
     */
    public Line2D getLine(long bar,
			  double x1, double y1,
			  double x2, double y2) {


	// get the intersection point of the bar and axis
	Point2D anchor = FiveFold.barNumToPoint(this, bar);

	// we know the angle is the angle of the sequence + 90 degrees
	double angle = rotation + 90;

	return getLineSegment(anchor, angle, x1, y1, x2, y2);
    }
    

    /**
     * <p>
     * Converts the sequence axis into a line suitable for drawing.
     * Clips the line to the bounding box provided.
     * </p>
     *
     * @param x1 The x of the first corner of the bounding box
     * @param y1 The y of the first corner of the bounding box
     * @param x2 The x of the second corner of the bounding box
     * @param y2 The y of the second corner of the bounding box
     *
     * @return Line2D Shape to draw the axis with
     */
    public Line2D getAxis(double x1, double y1,
			  double x2, double y2) {


	// use the center of the sequence as the anchor
	Point2D anchor = new Point2D.Double(centerX, centerY);

	return getLineSegment(anchor, rotation, x1, y1, x2, y2);
    }
    

    /**
     * <p>
     * Returns a line that goes through the point specified, with the
     * specified angle.  The returned line is also clipped against the
     * provided rectangle.  In the event that the line does not intersect
     * the rectangle, an empty (but not null) line is returned.
     * </p>
     *
     * @param anchor A point that the line passes through
     * @param angle The angle of rotation for the line
     * @param x1 The x of the first corner of the bounding box
     * @param y1 The y of the first corner of the bounding box
     * @param x2 The x of the second corner of the bounding box
     * @param y2 The y of the second corner of the bounding box
     *
     * @return Line2D The line segment that intersects the rectangle
     */
    public static Line2D getLineSegment(Point2D anchor, double angle,
					double x1, double y1,
					double x2, double y2) {

	// sides of the box
	double leftX, topY, rightX, bottomY;

	if (x1 < x2) {
	    leftX = x1;
	    rightX = x2;
	}
	else {
	    leftX = x2;
	    rightX = x1;
	}

	if (y1 < y2) {
	    topY = y2;
	    bottomY = y1;
	}
	else {
	    topY = y1;
	    bottomY = y2;
	}

	// Now solve for the intersection point with the box
	double cos = PenroseTiling.cos(angle);
	double sin = PenroseTiling.sin(angle);
       
	// sides to intersect with
	double leftY, topX, rightY, bottomX;

	leftY = (((leftX - anchor.getX()) / cos) * sin) + anchor.getY();
	topX = (((topY - anchor.getY()) / sin) * cos) + anchor.getX();
	rightY = (((rightX - anchor.getX()) / cos) * sin) + anchor.getY();
	bottomX = (((bottomY - anchor.getY()) / sin) * cos) + anchor.getX();

	double[] xs = new double[2];
	double[] ys = new double[2];
	int count = 0;

	if ( (leftY < topY) && (leftY > bottomY) ) {
	    xs[count] = leftX;
	    ys[count] = leftY;
	    count++;
	}

	if ( (topX < rightX) && (topX > leftX) ) {
	    xs[count] = topX;
	    ys[count] = topY;
	    count++;
	}

	if ( (rightY < topY) && (rightY > bottomY) ) {
	    xs[count] = rightX;
	    ys[count] = rightY;
	    count++;
	}

	if ( (bottomX < rightX) && (bottomX > leftX) ) {
	    xs[count] = bottomX;
	    ys[count] = bottomY;
	    count++;
	}
	
	if (count == 2) {
	    // We have intersected the rectangle
	    return new Line2D.Double(xs[0], ys[0], xs[1], ys[1]);
	}
	else {
	    // We have not intersected the rectangle correctly
	    return new Line2D.Double();
	}
    }
    

    /**
     * <p>
     * Returns a string representation of this Musical Sequence
     * </p>
     *
     * @return String This Musical Sequence as a String
     */
    public String toString() {
	// Default is to print out 10 bars of context starting at
	// Bar 0 (the "center")
	
	return this.toString(0, 10);
    }
    

    /**
     * <p>
     * Returns a string representation of a certain portion of this
     * Musical Sequence.
     * </p>
     *
     * @param start What bar to start with
     * @param range How many bars to print
     *
     * @return String This section of the Musical Sequence as a String
     */
    public String toString(long start, long range) {
	
	// Must figure out a way to get relative bar length here
	String temp = "\n| ";	// Bar 0 is forced, so print it here
	
	long lastHigh = 0;
	
	for (long i = 1; i <= range; i++)	{
	    
	    long shorter = findLowerPoint((start + i));
	    
	    //	Arbitrarily pick the Shortest possible first
	    
	    if ((shorter - lastHigh) == 1)	{
	  			// it is possible to make a short here, so do it
		
		temp = temp + "S ";
	    }
	    else	{
			  	//	we must make a long bar here
		
		temp = temp + "L ";
	    }
	    
	    lastHigh = shorter;
	    
	    if (isForced(start+i))	{
		temp = temp + "| ";
	    }
	    
	}	//	end for loop over bars
	
	temp = temp + "\n\nUpper Bound:\t" + findUpperIntercept() + "\nLower Bound:\t" + findLowerIntercept() + "\n";
	
	temp = temp + "\n\nUpper Bar " + upperX + " has y value " + upperY +
	    "\n\nLower Bar " + lowerX + " has y value " + lowerY;
	
	temp = temp + "\n\nCenter at:\t(" + centerX + " , " + centerY + ")\tRotation: " + rotation + " degrees\n";
	
	return temp;
    }
    

    /**
     * <p>
     * Parses command line arguments and sets the parameters of this
     * Musical Sequence.
     * </p>
     *
     * @param args The array of command line arguments
     */
    public void parseArgs (String args[])	{
	
	System.out.println("Getting center coordinates...");
	
	centerX = Double.parseDouble(args[0]);
	centerY = Double.parseDouble(args[1]);
	rotation = Double.parseDouble(args[2]);
	
	System.out.println("Forcing bars specified on command line...");
	
	for (int i = 3; i < args.length; i+=2)	{
	    long barNumTemp = Integer.parseInt(args[i]);
	    boolean barLenTemp = parseLen(args[i+1]);
	    String len = "";
	    if (barLenTemp)	{
		len = "farther";
	    }
	    else	{
		len = "closer";
	    }
	    
	    System.out.println("\tForcing bar " + barNumTemp + " to be " + len + "...");
	    this.force(barNumTemp, barLenTemp);
	}
	
    }


    /**
     * <p>
     * Parses a string representation of an interval length and converts
     * it into a boolean.
     * </p>
     *
     * @param len String representation of an interval length
     *
     * @return boolean The boolean form of the string
     */
    protected static boolean parseLen(String len)	{
	
	if (len.equals("l"))	{
	    return true;	
	}
	else if (len.equals("L"))	{
	    return true;
	}
	else if (len.equals("long"))	{
	    return true;
	}
	else if (len.equals("Long"))	{
	    return true;
	}
	else if (len.equals("LONG"))	{
	    return true;
	}
	else	{
	    return false;
	}
	
    }
    

    /**
     * <p>
     * Allows testing of the class on the command line.
     * </p>
     *
     * @param args Command line arguments
     */
    public static void main(String args[]) {

	/*
	System.out.println("Testing utility functions: ");

	double truncateTest;

	truncateTest = 1.1;
	System.out.println(truncateTest +
			   " open: " + truncateOpen(truncateTest) +
			   " closed: " + truncateClosed(truncateTest));

	truncateTest = 1.0;
	System.out.println(truncateTest +
			   " open: " + truncateOpen(truncateTest) +
			   " closed: " + truncateClosed(truncateTest));

	truncateTest = 0.0;
	System.out.println(truncateTest +
			   " open: " + truncateOpen(truncateTest) +
			   " closed: " + truncateClosed(truncateTest));

	truncateTest = -1.0;
	System.out.println(truncateTest +
			   " open: " + truncateOpen(truncateTest) +
			   " closed: " + truncateClosed(truncateTest));

	truncateTest = -1.1;
	System.out.println(truncateTest +
			   " open: " + truncateOpen(truncateTest) +
			   " closed: " + truncateClosed(truncateTest));

	*/
	
	MusicalSequence Ammann;

	Ammann = new MusicalSequence();

	Ammann.forceAtDistance(PenroseTiling.SHORT);
	System.out.println(Ammann.toString(0,3));

	Ammann = new MusicalSequence();
	Ammann.force(1,PenroseTiling.SHORTER);
	System.out.println(Ammann.toString(0,3));

	Ammann = new MusicalSequence();
	Ammann.forceAtDistance(PenroseTiling.LONG);
	System.out.println(Ammann.toString(0,3));

	Ammann = new MusicalSequence();
	Ammann.force(1,PenroseTiling.LONGER);
	System.out.println(Ammann.toString(0,3));
		
	Ammann = new MusicalSequence();
	Ammann.forceAtDistance(-PenroseTiling.SHORT);
	System.out.println(Ammann.toString(-2,3));

	Ammann = new MusicalSequence();
	Ammann.force(-1,PenroseTiling.SHORTER);
	System.out.println(Ammann.toString(-2,3));

	Ammann = new MusicalSequence();
	Ammann.forceAtDistance(-PenroseTiling.LONG);
	System.out.println(Ammann.toString(-2,3));

	Ammann = new MusicalSequence();
	Ammann.force(-1,PenroseTiling.LONGER);
	System.out.println(Ammann.toString(-2,3));
	
	Ammann.parseArgs(args);
	
	System.out.println(Ammann.toString());

	System.out.println("Hundredth Bar: " + Ammann.barToDouble(100));
	
	Ammann.doubleToBar(Ammann.barToDouble(1));
	Ammann.doubleToBar(Ammann.barToDouble(10));
	Ammann.doubleToBar(Ammann.barToDouble(1000));
	Ammann.doubleToBar(Ammann.barToDouble(1000000));
	
	System.exit(0);
	
    }
       
}
