// $Header: /opt/cvs/java/net/logn/penrose/PenroseApplet.java,v 1.30 2001/02/22 15:09:22 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.penrose;

//Properties support
import java.util.Properties;
import java.util.Enumeration;
// AWT Components
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
// AWT Events
import java.awt.event.KeyEvent;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
// Swing Components
import javax.swing.JApplet;
import javax.swing.JButton;
import javax.swing.JRadioButton;
import javax.swing.ButtonGroup;
import javax.swing.JCheckBox;
import javax.swing.Box;
import javax.swing.JLabel;
import javax.swing.JToolBar;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JMenuBar;
import javax.swing.KeyStroke;
import javax.swing.JColorChooser;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
// File reading (for properties)
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;


/**
 * <p>
 * <b>PenroseApplet</b> provides all the user interaction features for
 * the basic Penrose tiling software.  Since it runs in an applet, features
 * that would violate the applet security measures (such as saving files)
 * are not implemented.
 * </p>
 *
 * <p>
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.30 $
 *
 * Last Modified $Date: 2001/02/22 15:09:22 $ by $Author: jhealy $
 */
public class PenroseApplet extends JApplet {

    /** Penrose Tiling object to use for computation */
    protected static PenroseTiling penroseTiling;

    /** Properties object containing applet params and file properties */
    protected Properties userProperties = null;

    /** Text field at bottom of window; provides help and information */
    protected static JLabel statusBar;

    /** Panel that renders the Penrose Tiling */
    protected static PenrosePanel pPanel;

    /** The current mode that the applet is in */
    protected static int currentMode = 0;

    /** Array of descriptions for each mode */
    protected static String[] modes;

    /** Default mode constant */
    public static final int M_DEFAULT = 0;

    /** Mode constant for moving */
    public static final int M_MOVE = 1;

    /** Mode constant for zooming */
    public static final int M_ZOOM = 2;

    /** Mode constant for zooming in */
    public static final int M_ZOOM_IN = 3;

    /** Mode constant for zooming out */
    public static final int M_ZOOM_OUT = 4;

    // Buttons used on the tool panel
    protected PButton moveButton;
    protected PButton zoomButton;
    protected PButton zoomInButton;
    protected PButton zoomOutButton;

    // Buttons used to set vertex configs
    protected JButton aceButton;
    protected JButton deuceButton;
    protected JButton sunButton;
    protected JButton starButton;
    protected JButton jackButton;
    protected JButton queenButton;
    protected JButton kingButton;

    // Check box buttons
    protected JCheckBox axesButton;
    protected JCheckBox forcedButton;
    protected JCheckBox unforcedButton;
    protected JCheckBox intersectionsButton;
    protected JCheckBox kitesButton;
    protected JCheckBox dartsButton;
    protected JButton refreshButton;
    protected JButton recomputeButton;
    protected JButton forceButton;
    protected JCheckBox fillTilesButton;
    protected JCheckBox markCenterButton;
    protected JCheckBox printColorButton;
    protected JRadioButton tileButton;
    protected JRadioButton rhombButton;


    // Initialize static variables
    static {

	penroseTiling = new PenroseTiling();

	modes = new String[6];

	modes[M_DEFAULT] = "Penrose Empire Applet: http://www.logn.net/penrose/";
	modes[M_MOVE] = "Click and drag to reposition viewing canvas";
	modes[M_ZOOM] = "Click to zoom";
	modes[M_ZOOM_IN] = "Click to zoom in 2x";
	modes[M_ZOOM_OUT] = "Click to zoom out 2x";

    }
    

    /**
     * <p>
     * Constructor.  Should not be called by applications.
     * </p>
     * 
     */
    public PenroseApplet() {
    }


    /**
     * <p>
     * Constructor.  Only called when run as an application.  Adds a menu
     * to the applet with addition non-applet functionality.
     * </p>
     * 
     * @param params The Properties object to get the parameters from
     */
    public PenroseApplet(Properties params) {

	userProperties = params;
	if (userProperties == null) {
	    // properties cannot be null if called as an application
	    userProperties = new Properties();
	}

         //Create the menu bar.
	JMenuBar menuBar = new JMenuBar();
	JMenuItem menuItem;
	JMenu menu, submenu;

	//Build the first menu.
	menu = new JMenu("File");
	menu.setMnemonic(KeyEvent.VK_F);
	menu.getAccessibleContext().setAccessibleDescription("File Operations");
	menuBar.add(menu);
	
	
	// open saved penrose tiling
	menuItem = new JMenuItem("Open Tiling Configuration");
	menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, ActionEvent.ALT_MASK));
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    
		    int answer =
			JOptionPane.showConfirmDialog(
						      PenroseApplet.this,
						      "Do you want to destroy the current empire settings?",
						      "Are you sure?",
						      JOptionPane.YES_NO_OPTION);
		    
		    if (answer == JOptionPane.YES_OPTION) {
			
			JFileChooser fileChooser = new JFileChooser();
			
			int returnVal = fileChooser.showOpenDialog(PenroseApplet.this);
			
			if (returnVal == JFileChooser.APPROVE_OPTION) {
			    File file = fileChooser.getSelectedFile();
			    
			    try {
				penroseTiling.loadConfiguration(file);
			    }
			    catch (IOException ioe) {
				statusBar.setText("Error Opening " + file.getName() + "!");
				System.err.println("Error Opening " + file.getName() + ": " + ioe.getMessage());
			    }
			} else {
			    statusBar.setText("Open command cancelled by user.");
			}
		    
			pPanel.redraw();
		    }
		}
	    });
	menu.add(menuItem);


	// save current penrose tiling
	menuItem = new JMenuItem("Save Tiling Configuration");
	menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, ActionEvent.ALT_MASK));
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    
		    JFileChooser fileChooser = new JFileChooser();
		    
		    int returnVal = fileChooser.showSaveDialog(PenroseApplet.this);
		    		    
		    if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();

			try {
			    penroseTiling.saveConfiguration(file);

			    statusBar.setText("Saved to " + file.getName());
			}
			catch (IOException ioe) {
			    statusBar.setText("Error Saving " + file.getName() + "!");
			    System.err.println("Error Saving " + file.getName() + ": " + ioe.getMessage());
			}
		    } else {
			statusBar.setText("Save command cancelled by user.");
		    }
		}
	    });
	menu.add(menuItem);


	menu.addSeparator();

	// open user properties
	menuItem = new JMenuItem("Open System Properties...");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    
		    JFileChooser fileChooser = new JFileChooser();

		    int returnVal = fileChooser.showOpenDialog(PenroseApplet.this);
		    
		    if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();

			Properties userProps = new Properties();

			try {
			    FileInputStream inStream = new FileInputStream(file);
			    userProps.load(inStream);
			    penroseTiling.loadProperties(userProps);
			}
			catch (IOException ioe) {
			    statusBar.setText("Error Loading " + file.getName() + "!");
			    System.err.println("Error Loading " + file.getName() + ": " + ioe.getMessage());
			}
		    } else {
			statusBar.setText("Load command cancelled by user.");
		    }
		}
	    });
	menu.add(menuItem);


	// save user properties
	menuItem = new JMenuItem("Save System Properties...");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    
		    JFileChooser fileChooser = new JFileChooser();

		    int returnVal = fileChooser.showSaveDialog(PenroseApplet.this);
		    if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();

			try {
			    penroseTiling.saveProperties(file);
			    statusBar.setText("Saved to " + file.getName());
			}
			catch (IOException ioe) {
			    statusBar.setText("Error Saving " + file.getName() + "!");
			    System.err.println("Error Saving " + file.getName() + ": " + ioe.getMessage());
			}
		    } else {
			statusBar.setText("Save command cancelled by user.");
		    }
		}
	    });
	menu.add(menuItem);


	// export submenu
	menu.addSeparator();
	submenu = new JMenu("Export Rendering");
	submenu.setMnemonic(KeyEvent.VK_E);
	

	menuItem = new JMenuItem("PNG (requires JAI)");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    
		    JFileChooser fileChooser = new JFileChooser();

		    int returnVal = fileChooser.showSaveDialog(PenroseApplet.this);
		    if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();

			try {
			    penroseTiling.exportPNG(file, pPanel.getWidth(),
						    pPanel.getHeight());

			    statusBar.setText("Saved to " + file.getName());
			}
			catch (IOException ioe) {
			    statusBar.setText("Error Exporting to " + file.getName() + "!");
			    System.err.println("Error Exporting to " + file.getName() + ": " + ioe.getMessage());
			}
		    } else {
			statusBar.setText("Export command cancelled by user.");
		    }
		}
	    });
	submenu.add(menuItem);

	menuItem = new JMenuItem("EPS");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    
		    JFileChooser fileChooser = new JFileChooser();

		    int returnVal = fileChooser.showSaveDialog(PenroseApplet.this);
		    if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();

			try {
			    penroseTiling.exportEPS(file);

			    statusBar.setText("Saved to " + file.getName());
			}
			catch (IOException ioe) {
			    statusBar.setText("Error Exporting to " + file.getName() + "!");
			    System.err.println("Error Exporting to " + file.getName() + ": " + ioe.getMessage());
			}
		    } else {
			statusBar.setText("Export command cancelled by user.");
		    }
		}
	    });
	submenu.add(menuItem);

	menuItem = new JMenuItem("PDF");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    
		    JFileChooser fileChooser = new JFileChooser();

		    int returnVal = fileChooser.showSaveDialog(PenroseApplet.this);
		    if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();

			try {
			    penroseTiling.exportPDF(file);

			    statusBar.setText("Saved to " + file.getName());
			}
			catch (IOException ioe) {
			    statusBar.setText("Error Exporting to " + file.getName() + "!");
			    System.err.println("Error Exporting to " + file.getName() + ": " + ioe.getMessage());
			}
		    } else {
			statusBar.setText("Export command cancelled by user.");
		    }
		}
	    });
	submenu.add(menuItem);

	menu.add(submenu);

	menu.addSeparator();

	// print
	menuItem = new JMenuItem("Print");
	menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P, ActionEvent.ALT_MASK));
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {

		    PenroseApplet.penroseTiling.print();
		    
		}
	    });
	menu.add(menuItem);

	menu.addSeparator();

	// quit
	menuItem = new JMenuItem("Quit", KeyEvent.VK_Q);
	menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q, ActionEvent.ALT_MASK));
	menuItem.getAccessibleContext().setAccessibleDescription("Quits the Program");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    System.exit(0);
		}
	    });
	menu.add(menuItem);
	
	//Build second menu in the menu bar.
	menu = new JMenu("Color");
	menu.setMnemonic(KeyEvent.VK_C);
	menu.getAccessibleContext().setAccessibleDescription("Set the colors used by the software");
	menuBar.add(menu);

	menuItem = new JMenuItem("Background");
	menuItem.getAccessibleContext().setAccessibleDescription("Sets the background color");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    penroseTiling.backgroundColor = JColorChooser.showDialog(PenroseApplet.this,
							  "Choose Background Color",
							  penroseTiling.backgroundColor);
		}
	    });
	menu.add(menuItem);

	menuItem = new JMenuItem("Axes");
	menuItem.getAccessibleContext().setAccessibleDescription("Sets the axes color");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    penroseTiling.axesColor= JColorChooser.showDialog(PenroseApplet.this,
							  "Choose Axes Color",
							  penroseTiling.axesColor);
		}
	    });
	menu.add(menuItem);

	menuItem = new JMenuItem("Forced Bars");
	menuItem.getAccessibleContext().setAccessibleDescription("Sets the forced bar color");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    penroseTiling.forcedColor= JColorChooser.showDialog(PenroseApplet.this,
							  "Choose Forced Bar Color",
							  penroseTiling.forcedColor);
		}
	    });
	menu.add(menuItem);

	menuItem = new JMenuItem("Unforced Bars");
	menuItem.getAccessibleContext().setAccessibleDescription("Sets the unforced bar color");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    penroseTiling.unforcedColor= JColorChooser.showDialog(PenroseApplet.this,
							  "Choose Unforced Bar Color",
							  penroseTiling.unforcedColor);
		}
	    });
	menu.add(menuItem);

	menuItem = new JMenuItem("Intersections");
	menuItem.getAccessibleContext().setAccessibleDescription("Sets the intersections color");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    penroseTiling.intersectionsColor= JColorChooser.showDialog(PenroseApplet.this,
							  "Choose Intersections Color",
							  penroseTiling.intersectionsColor);
		}
	    });
	menu.add(menuItem);

	menuItem = new JMenuItem("Kites");
	menuItem.getAccessibleContext().setAccessibleDescription("Sets the Kite Color");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    penroseTiling.kitesColor= JColorChooser.showDialog(PenroseApplet.this,
							  "Choose Kite Color",
							  penroseTiling.kitesColor);
		}
	    });
	menu.add(menuItem);

	menuItem = new JMenuItem("Darts");
	menuItem.getAccessibleContext().setAccessibleDescription("Sets the Dart Color");
	menuItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    penroseTiling.dartsColor= JColorChooser.showDialog(PenroseApplet.this,
							  "Choose Dart Color",
							  penroseTiling.dartsColor);
		}
	    });
	menu.add(menuItem);

	// set the menu for the applet
	this.setJMenuBar(menuBar);
	
    }


    /**
     * <p>
     * Initializes the applet and prepares it for running.
     * </p>
     * 
     */
    public void init(){

	if (userProperties == null) {
	    userProperties = new Properties();

	    // If the properties are null, then we have been called as
	    // an applet, so we have to get the applet parameters
	    
	    // list of parameter values to read over
	    String[] keys = {"debug", "llx", "lly", "urx", "ury",
			     "drawAxes", "drawForced", "drawUnforced",
			     "drawIntersections", "drawKites", "drawDarts",
			     "markCenter", "fillTiles", "exportColor",
			     "backgroundColor", "axesColor", "forcedColor",
			     "unforcedColor", "intersectionsColor",
			     "kitesColor", "dartsColor", "fillTransparency"
	    };
	    
	    for (int i = 0; i < keys.length; i++) {
		String value = getParameter(keys[i]);
		if (value != null) {
		    userProperties.setProperty(keys[i], value);
		}
	    }
	    
	}
	
	// finally, load the properties
	penroseTiling.loadProperties(userProperties);

        getContentPane().setLayout(new BorderLayout());

        pPanel = new PenrosePanel();
        pPanel.setBackground(penroseTiling.backgroundColor);
        getContentPane().add(pPanel);

        //Create the top toolbar.
        JToolBar toolBar = new JToolBar(JToolBar.HORIZONTAL);
        addButtons(toolBar);
	getContentPane().add("North", toolBar);

        //Create the checkbox toolbar.
        JToolBar checkBoxBar = new JToolBar(JToolBar.VERTICAL);
        addCheckBoxes(checkBoxBar);
	getContentPane().add("East", checkBoxBar);

        statusBar = new JLabel(modes[M_DEFAULT]);
        getContentPane().add("South", statusBar);

    }


    /**
     * <p>
     * Adds all the tools to the specified tool bar.
     * </p>
     * 
     * @param toolBar The tool bar to add tools to
     */
    protected void addButtons(JToolBar toolBar) {

	ButtonListener buttonListener = new ButtonListener();

	JLabel toolsLabel = new JLabel("Tools:");
	toolBar.add(toolsLabel);
	toolBar.addSeparator(new Dimension(10, 10));

        // Move
        moveButton = new PButton("icons/move");
	moveButton.setMode(M_MOVE);
	moveButton.setMnemonic(KeyEvent.VK_M);
        moveButton.setToolTipText("Move viewpoint");
        moveButton.addActionListener(buttonListener);
        toolBar.add(moveButton);
	
        // Zoom In
        zoomInButton = new PButton("icons/zoomIn");
	zoomInButton.setMode(M_ZOOM_IN);
	zoomInButton.setMnemonic(KeyEvent.VK_EQUALS);
        zoomInButton.setToolTipText("Zoom in 2x");
        zoomInButton.addActionListener(buttonListener);
        toolBar.add(zoomInButton);

	// Zoom Arbitrary
	zoomButton = new PButton("icons/zoom");
	zoomButton.setMode(M_ZOOM);
	zoomButton.setMnemonic(KeyEvent.VK_Z);
	zoomButton.setToolTipText("Zoom By An Arbitrary Amount");
	zoomButton.addActionListener(buttonListener);
	zoomButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    double newZoom = pPanel.getZoom();
		    
		    ZoomDialog zoomDialog = new ZoomDialog(newZoom);
		    zoomDialog.pack();
		    
		    zoomDialog.setVisible(true);
		    newZoom = zoomDialog.getValidatedZoom();
		    pPanel.setZoom(newZoom);
		}
	    });
	toolBar.add(zoomButton);
	
        // Zoom Out
        zoomOutButton = new PButton("icons/zoomOut");
	zoomOutButton.setMode(M_ZOOM_OUT);
	zoomOutButton.setMnemonic(KeyEvent.VK_MINUS);
        zoomOutButton.setToolTipText("Zoom Out 2x");
        zoomOutButton.addActionListener(buttonListener);
        toolBar.add(zoomOutButton);


	toolBar.add(new Box.Filler(new Dimension(20, 10),
				   new Dimension(50, 10),
				   new Dimension(150, 150)));

	JLabel vertexLabel = new JLabel("Vertex Configurations:");
	toolBar.add(vertexLabel);
	toolBar.addSeparator(new Dimension(10, 10));

	// ace
	aceButton = new JButton(classIcon("icons/ace.gif"));
	aceButton.setMnemonic(KeyEvent.VK_1);
	aceButton.setToolTipText("Generate Ace Configuration");
	aceButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    newVertex(PenroseTiling.ACE);
		}
	    });
	toolBar.add(aceButton);

	// deuce
	deuceButton = new JButton(classIcon("icons/deuce.gif"));
	deuceButton.setMnemonic(KeyEvent.VK_2);
	deuceButton.setToolTipText("Generate Deuce Configuration");
	deuceButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    newVertex(PenroseTiling.DEUCE);
		}
	    });
	toolBar.add(deuceButton);

	// sun
	sunButton = new JButton(classIcon("icons/sun.gif"));
	sunButton.setMnemonic(KeyEvent.VK_3);
	sunButton.setToolTipText("Generate Sun Configuration");
	sunButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    newVertex(PenroseTiling.SUN);
		}
	    });
	toolBar.add(sunButton);

	// star
	starButton = new JButton(classIcon("icons/star.gif"));
	starButton.setMnemonic(KeyEvent.VK_4);
	starButton.setToolTipText("Generate Star Configuration");
	starButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    newVertex(PenroseTiling.STAR);
		}
	    });
	toolBar.add(starButton);

	// jack
	jackButton = new JButton(classIcon("icons/jack.gif"));
	jackButton.setMnemonic(KeyEvent.VK_5);
	jackButton.setToolTipText("Generate Jack Configuration");
	jackButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    newVertex(PenroseTiling.JACK);
		}
	    });
	toolBar.add(jackButton);

	// queen
	queenButton = new JButton(classIcon("icons/queen.gif"));
	queenButton.setMnemonic(KeyEvent.VK_6);
	queenButton.setToolTipText("Generate Queen Configuration");
	queenButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    newVertex(PenroseTiling.QUEEN);
		}
	    });
	toolBar.add(queenButton);

	// king
	kingButton = new JButton(classIcon("icons/king.gif"));
	kingButton.setMnemonic(KeyEvent.VK_7);
	kingButton.setToolTipText("Generate King Configuration");
	kingButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    newVertex(PenroseTiling.KING);
		}
	    });
	toolBar.add(kingButton);

    }


    /**
     * <p>
     * Asks the user if they want to use a new vertex configuration, and
     * does so if the assent.
     * </p>
     * 
     * @param configuration The configuration to use
     */
    protected void newVertex(int configuration) {
	int answer =
	    JOptionPane.showConfirmDialog(
					  PenroseApplet.this,
					  "Do you want to destroy the current empire settings?",
					  "Are you sure?",
					  JOptionPane.YES_NO_OPTION);
	
	if (answer == JOptionPane.YES_OPTION) {
	    penroseTiling.setInitialConfiguration(configuration);
	    pPanel.redraw();
	}
    }
    

    /**
     * <p>
     * Adds all the check box buttons to the specified tool bar
     * </p>
     * 
     * @param toolBar The tool bar to add tools to
     */
    protected void addCheckBoxes(JToolBar toolBar) {

	ButtonListener buttonListener = new ButtonListener();

        // drawAxes
	axesButton = new JCheckBox("Axes");
	axesButton.setMnemonic(KeyEvent.VK_A);
	axesButton.setSelected(penroseTiling.drawAxes());
        axesButton.setToolTipText("Axes will be drawn");
        axesButton.addItemListener(buttonListener);
        toolBar.add(axesButton);
	
        // drawForced
	forcedButton = new JCheckBox("Forced Bars");
	forcedButton.setMnemonic(KeyEvent.VK_F);
	forcedButton.setSelected(penroseTiling.drawForced());
        forcedButton.setToolTipText("Forced bars will be drawn");
        forcedButton.addItemListener(buttonListener);
        toolBar.add(forcedButton);
	
        // drawUnforced
	unforcedButton = new JCheckBox("Unforced Bars");
	unforcedButton.setMnemonic(KeyEvent.VK_U);
	unforcedButton.setSelected(penroseTiling.drawUnforced());
        unforcedButton.setToolTipText("Unforced bars will be drawn");
        unforcedButton.addItemListener(buttonListener);
        toolBar.add(unforcedButton);
	
        // drawIntersections
	intersectionsButton = new JCheckBox("Intersections");
	intersectionsButton.setMnemonic(KeyEvent.VK_I);
	intersectionsButton.setSelected(penroseTiling.drawIntersections());
        intersectionsButton.setToolTipText("Intersections will be drawn");
        intersectionsButton.addItemListener(buttonListener);
        toolBar.add(intersectionsButton);
	
        // drawKites
	kitesButton = new JCheckBox("Kites");
	kitesButton.setMnemonic(KeyEvent.VK_K);
	kitesButton.setSelected(penroseTiling.drawKites());
        kitesButton.setToolTipText("Kites will be drawn");
        kitesButton.addItemListener(buttonListener);
        toolBar.add(kitesButton);
	
        // drawDarts
	dartsButton = new JCheckBox("Darts");
	dartsButton.setMnemonic(KeyEvent.VK_D);
	dartsButton.setSelected(penroseTiling.drawDarts());
        dartsButton.setToolTipText("Darts will be drawn");
        dartsButton.addItemListener(buttonListener);
        toolBar.add(dartsButton);
	
        // Refresh
        refreshButton = new JButton("Refresh Empire");
	refreshButton.setMnemonic(KeyEvent.VK_R);
        refreshButton.setToolTipText("Refresh Empire (without finding new forcings)");
        refreshButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
		pPanel.redraw();
            }
        });
        toolBar.add(refreshButton);

        // Recompute
        recomputeButton = new JButton("Recompute Empire");
	recomputeButton.setMnemonic(KeyEvent.VK_C);
        recomputeButton.setToolTipText("Recompute Empire (may take a LONG time!)");
        recomputeButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
		pPanel.recompute();
		pPanel.redraw();
            }
        });
        toolBar.add(recomputeButton);

	toolBar.add(new Box.Filler(new Dimension(10, 10),
				   new Dimension(20, 40),
				   new Dimension(40, 80)));

	// Force Arbitrary
	forceButton = new JButton("Force Arbitrary");
	forceButton.setMnemonic(KeyEvent.VK_N);
	forceButton.setToolTipText("Force New Arbitrary Bars");
	forceButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {

		    ForceBarFrame newBars =
			new ForceBarFrame(PenroseApplet.this);
		    
		    newBars.setSize(new Dimension(400, 150));
		    newBars.show();
		}
	    });
	toolBar.add(forceButton);

	toolBar.add(new Box.Filler(new Dimension(10, 10),
				   new Dimension(20, 40),
				   new Dimension(40, 80)));

        // fillTiles
	fillTilesButton = new JCheckBox("Fill Tiles");
	fillTilesButton.setSelected(penroseTiling.fillTiles());
        fillTilesButton.setToolTipText("Tiles will be filled");
        fillTilesButton.addItemListener(buttonListener);
        toolBar.add(fillTilesButton);
	
        // markCenter
	markCenterButton = new JCheckBox("Mark Center");
	markCenterButton.setSelected(penroseTiling.markCenter());
        markCenterButton.setToolTipText("Set whether to mark the center of the image");
        markCenterButton.addItemListener(buttonListener);
        toolBar.add(markCenterButton);
	
        // printColor
	printColorButton = new JCheckBox("Export in Color");
	printColorButton.setSelected(penroseTiling.exportColor());
        printColorButton.setToolTipText("Set whether to print and export in color");
        printColorButton.addItemListener(buttonListener);
        toolBar.add(printColorButton);

	/*
	  Disable tiles v rhombs choice until we have rhombs working
	// tiles vs. rhombs
	tileButton = new JRadioButton("Tiles");
	tileButton.setSelected(penroseTiling.tilesNotRhombs());
	tileButton.setToolTipText("Show Tiles as Penrose Tiles");
        tileButton.addItemListener(buttonListener);
	toolBar.add(tileButton);

	rhombButton = new JRadioButton("Rhombi");
	rhombButton.setSelected(!penroseTiling.tilesNotRhombs());
	rhombButton.setToolTipText("Show Tiles as Penrose Rhombi");
        rhombButton.addItemListener(buttonListener);
	toolBar.add(rhombButton);

	// Group the radio buttons.
	ButtonGroup group = new ButtonGroup();
	group.add(tileButton);
	group.add(rhombButton);
	*/

    }


    /**
     * Loads the icon from the class path
     *
     * @return ImageIcon The icon requested
     */
    public static ImageIcon classIcon(String name) {
	
	ImageIcon icon = null;
	
	java.net.URL iconURL = ClassLoader.getSystemResource("net/logn/penrose/" + name);
	if (iconURL != null) {
	    icon = new ImageIcon(iconURL);
	}
	
	return icon;
    }
    

    class PButton extends JButton {

	protected int mode = 0;
	protected Icon inactive = null;

	/**
	 * Creates a button with an icon.
	 *
	 * @param icon  the Icon image to display on the button
	 */
	public PButton(Icon icon) {
	    super(icon);
	}


	/**
	 * Creates a button with an icon loaded from the given string.
	 *
	 * @param icon  the Icon image to display on the button
	 */
	public PButton(String icon) {
	    super(PenroseApplet.classIcon(icon + ".gif"));
	    inactive = PenroseApplet.classIcon(icon + "Selected.gif");
	}
	
	/*
	 * Sets the mode of the button.
	 *
	 * @param newMode The mode to associate with the button
	 */
	public void setMode(int newMode) {
	    mode = newMode;
	}

	/**
	 * Gets the current mode of the button
	 *
	 * @return int The mode of the button
	 */
	public int getMode() {
	    return mode;
	}

	/**
	 * Gets the current mode of the button
	 *
	 * @return int The mode of the button
	 */
	public void swap() {
	    if (inactive != null) {
		Icon temp = this.getIcon();
		this.setIcon(inactive);
		inactive = temp;
	    }
	}

    }


/**
 * <p>
 * <b>ButtonListener</b> provides a listener that will act on the events
 * of certain buttons being pushed.
 * </p>
 */
    class ButtonListener implements ItemListener, ActionListener {

	PButton active = null;

	/**
	 * Handles a buttons state changing.  This is used to detect
	 * when a checkbox has been checked or unchecked.
	 */
        public void itemStateChanged(ItemEvent e) {
            Object source = e.getItemSelectable();
	    
	    boolean state = false;

	    if (e.getStateChange() == ItemEvent.SELECTED) {
		state = true;
	    }

	    if (source == axesButton) {
		penroseTiling.drawAxes(state);
	    }
	    else if (source == forcedButton) {
		penroseTiling.drawForced(state);
	    }
	    else if (source == unforcedButton) {
		penroseTiling.drawUnforced(state);
	    }
	    else if (source == intersectionsButton) {
		penroseTiling.drawIntersections(state);
	    }
	    else if (source == kitesButton) {
		penroseTiling.drawKites(state);
	    }
	    else if (source == dartsButton) {
		penroseTiling.drawDarts(state);
	    }
	    else if (source == fillTilesButton) {
		penroseTiling.fillTiles(state);
	    }
	    else if (source == markCenterButton) {
		penroseTiling.markCenter(state);
	    }
	    else if (source == printColorButton) {
		penroseTiling.exportColor(state);
	    }
	    else if (source == tileButton) {
		penroseTiling.tilesNotRhombs(state);
	    }
	    else if (source == rhombButton) {
		penroseTiling.tilesNotRhombs(!state);
	    }
        }

	
	/**
	 * Handles a button completing its action.  This is used to
	 * handle a user clicking on a regular button.
	 */
        public void actionPerformed(ActionEvent e) {
            PButton source = (PButton)e.getSource();
	    int mode = source.getMode();

	    currentMode = mode;
	    statusBar.setText(modes[currentMode]);

	    if (active != null) {
		active.swap();
	    }

	    active = source;
	    active.swap();

        }
    }


    /**
     * <p>
     * Starts the applet from a shell prompt.  Simply a wrapper around
     * the applet code.
     * </p>
     * 
     * @param args Command-line arguments
     */
    public static void main(String args[]) {
        JFrame wrapper = new JFrame("Penrose Applet With Extensions");

	Properties parameters = new Properties();

	// If the user provided an options file, use it
	if (args.length > 0) {
	    if (args.length > 1) {
		System.out.println("PenroseApplet takes one optional argument.  Ignoring other arguments...");
	    }

	    // get the properties file
	    try {
		FileInputStream propertiesIn = new FileInputStream(args[0]);
		parameters.load(propertiesIn);
	    }
	    catch (IOException ioe) {
		System.err.println("Error reading properties file '" +
				   args[0] + "'.\n\nUsing Defaults...");
	    }
	}


	// Add code that quits the applet when the window closes
        wrapper.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {System.exit(0);}
        });

	// Add the applet to this shell window
        JApplet applet = new PenroseApplet(parameters);
        wrapper.getContentPane().add("Center", applet);

	// Start the applet
        applet.init();
        wrapper.pack();
        wrapper.setSize(new Dimension(640, 480));
        wrapper.show();
    }

    
}

