// $Header: /opt/cvs/java/net/logn/penrose/PenrosePanel.java,v 1.16 2001/02/15 20:42:35 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.penrose;

// AWT Components
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Shape;
// AWT Events
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseEvent;
// Image (for buffering)
import java.awt.image.BufferedImage;
// Swing Components
import javax.swing.JPanel;
// Geometrical help
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.NoninvertibleTransformException;


/**
 * <p>
 * <b>PenrosePanel</b> is an extension of JPanel.  It will perform all of the
 * GUI-oriented tasks for the Penrose Tiling software.
 * </p>
 *
 * <p>
 * This class allows the user to manipulate the view of a Penrose Tiling.
 * Because Penrose Tilings are expensive to render, it only reconstructs
 * the rendering of the tiling when explicitly told to.  The rest of the
 * time, it uses a buffered "picture" of the tiling which can be easily
 * moved and scaled.  Once the user has the tiling at the desired 
 * position and scale, it can be re-rendered to fill the window.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.16 $
 *
 * Last Modified $Date: 2001/02/15 20:42:35 $ by $Author: jhealy $
 */
public class PenrosePanel extends JPanel
    implements MouseListener, MouseMotionListener {

    /** Contains the buffered version of the tiling */
    protected BufferedImage viewport;

    /** Holds the transformations made by the user before they are
	rendered in the final image */
    protected AffineTransform previewTransform;

    /** Holds the transformation needed to map the viewable image into
	the window */
    protected AffineTransform viewportTransform;

    /** Current width of the drawing window */
    protected int appWidth;

    /** Current height of the drawing window */
    protected int appHeight;

    /** Current X-coordinate that should be at the center of the window */
    protected double centerX;

    /** Current Y-coordinate that should be at the center of the window */
    protected double centerY;

    /** Current scale applied to the drawing window */
    protected double scale;

    /** Current zoom factor to use when user clicks */
    protected double zoomFactor;

    // Holds the click coordinates of the mouse
    private int pressX, pressY;
    // Holds the release coordinates of the mouse
    private int releaseX, releaseY;

    // These points hold the clicks, but in the coordinate system of the
    // Penrose tiles (which is often different from the screen coordinates)
    private static final Point2D.Double zero = new Point2D.Double(0.0, 0.0);
    private Point2D.Double press = zero;
    private Point2D.Double release = zero;
    private Point2D.Double center = zero;

    /**
     * <p>
     * Creates a new PenrosePanel, centered at 0,0 with a scale of 1.
     * </p>
     * 
     */   
    public PenrosePanel() {
	setBackground(PenroseTiling.backgroundColor);
	addMouseMotionListener(this);
	addMouseListener(this);

	// Initialize the transformations to the identity transformation
	viewportTransform = new AffineTransform();
	previewTransform = new AffineTransform();

	// set the scale to 100
	scale = 100;

	// set the zoom factor to 1 (default)
	zoomFactor = 1;
	
	// need to initialize the image to display
	viewport = new BufferedImage(1, 1, BufferedImage.TYPE_INT_RGB);
    }


    /**
     * <p>
     * Recalculates the size of the image based on the size of the
     * applet.  Used when the applet changes size.
     * </p>
     * 
     */   
    protected void recalc() {

	// Construct a new transformation that will map our rendering
	// into the image to be displayed.
	viewportTransform = new AffineTransform();

	// Get the current width and height of the window
	appWidth = getWidth();
	appHeight = getHeight();

	// Update the tiling's coordinates
	PenroseApplet.penroseTiling.setDimensions((appWidth / scale),
						  (appHeight / scale));
	PenroseApplet.penroseTiling.setOffset(center.getX(), center.getY());

	// scale the image based on our current scale factor
	viewportTransform.scale(scale, scale);

	// now we need to center the image
	viewportTransform.translate( (appWidth / (2 * scale)), (appHeight / (2 * scale)));
	viewportTransform.translate(-center.getX(), -center.getY());


    }

    /**
     * <p>
     * Recomputes the penrose empire.  This may take a VERY long time, and
     * so must be explicitly called
     * </p>
     * 
     */   
    public void recompute() {

	recalc();

	Graphics2D renderPreview = (Graphics2D)this.getGraphics();
	renderPreview.setTransform(viewportTransform);

	PenroseApplet.penroseTiling.recomputeEmpire(renderPreview);
    }


    /**
     * <p>
     * Re-renders the image to the screen.  This may be an expensive
     * method to call, so it should only be called when a complete
     * redraw is requested by the user.
     * </p>
     * 
     */   
    public void redraw() {

	recalc();
	
	viewport = new BufferedImage(appWidth, appHeight, BufferedImage.TYPE_INT_RGB);
	// Get the graphics representation of the image
	Graphics2D graphics = viewport.createGraphics();

	// Clear the image
	graphics.setBackground(PenroseTiling.backgroundColor);
	graphics.clearRect(0, 0, appWidth, appHeight);

	graphics.setTransform(viewportTransform);
	PenroseApplet.penroseTiling.renderTiling(graphics);

	// reset the preview window to it's regular position
	previewTransform = new AffineTransform();
	releaseX = pressX;
	releaseY = pressY;

	// paint the window
	repaint();

    }


    /**
     * <p>
     * Draws the provided shape to the screen, over everything else.  Can be
     * used the preview objects to the screen without adding them to the
     * tiling.
     * </p>
     * 
     * @param shape The Shape to draw to the screen
     * @param color The color to draw in
     * @param fill Whether or not to fill the shape
     */   
    public void paintShape(Shape shape, Color color, boolean fill) {

	recalc();
	
	viewport = new BufferedImage(appWidth, appHeight, BufferedImage.TYPE_INT_RGB);
	// Get the graphics representation of the image
	Graphics2D graphics = viewport.createGraphics();

	// Clear the image
	graphics.setBackground(PenroseTiling.backgroundColor);
	graphics.clearRect(0, 0, appWidth, appHeight);

	graphics.setTransform(viewportTransform);
	PenroseApplet.penroseTiling.renderTiling(graphics);

	// reset the preview window to it's regular position
	previewTransform = new AffineTransform();
	releaseX = pressX;
	releaseY = pressY;


	graphics.setColor(color);
	// Now add the shape
	if (fill) {
	    graphics.fill(shape);
	}
	else {
	    graphics.draw(shape);
	}

	// paint the window
	repaint();

    }


    /**
     * <p>
     * This is a "lite" redrawing of the window.  It essentially
     * redraws the pre-rendered image on the screen, based on where
     * the user clicks.  However, the rendering may not fill the screen
     * nor be at sharpest quality.  This method is intened to provide a
     * rough update of the screen to aid in positioning so that the more
     * expensive redraw() method gets called less often.
     * </p>
     * 
     * @param g The Graphics component to draw to
     */   
    public void paintComponent(Graphics g){

        super.paintComponent(g);
        Graphics2D g2 = (Graphics2D)g;

	// Draw the preview into the window, using the
	// transformation to put it in the correct position
 	g2.drawRenderedImage(viewport, previewTransform);

    }


    /**
     * <p>
     * Sets the current zoom factor.  Factors greater than 1 result in 
     * zooming in; factors less than 1 result in zooming out. If the 
     * factor is exactly 1, then the zoom is restored to its default amount.
     * </p>
     *
     * @param factor The desired zoom factor
     */
    public void setZoom(double factor) {

	if ( (factor == 1.0) && (scale != 1.0) ) {
	    zoomFactor = (1/scale);
	}
	else {
	    zoomFactor = factor;
	}
    }


    /**
     * <p>
     * Returns the current zoom factor
     * </p>
     *
     * @return double The current zoom factor
     */
    public double getZoom() {

	return zoomFactor;
    }


    /**
     * <p>
     * Zooms the screen by the amount specified in the zoom instance
     * variable.
     * </p>
     */
    public void zoom() {

	// when we change the size of the preview image, it is no longer
	// offset correctly.  We therefore store the current offset
	// so that it can be multiplied by the zoom factor
	double correctionX = previewTransform.getTranslateX();
	double correctionY = previewTransform.getTranslateY();

	// determine what preview coordinate lies at the center of the
	// applet window.  The division is to get the coordinates in
	// native screen scale, not in the scale of the preview.
	// This "center" coordinate is the center of the window's distance
	// from the (0, 0) coordinate of the preview image.
	double centerX = ( (appWidth / 2) - correctionX ) / previewTransform.getScaleX();
	double centerY = ( (appHeight / 2) - correctionY ) / previewTransform.getScaleY();

	// now set the offset of the window.  This computation is just a litte
	// algebra; a point in the image x units from 0 will move to
	// (factor * x) units away.  So we need to subtract this shift, or:
	// x - (factor * x) which reduces to x(1 - factor)
	double offsetX = (1 - zoomFactor) * centerX;
	double offsetY = (1 - zoomFactor) * centerY;

	// now apply this new centering
	previewTransform.translate(offsetX, offsetY);

	// set the new scale
	scale = scale * zoomFactor;

	// set the new scale on the preview (must do this after the offset,
	// or else the offset will be applied in the new scale factor and
	// be incorrect
	previewTransform.scale(zoomFactor, zoomFactor);

	// whew!  now repaint
       	repaint();
    }


    /**
     * <p>Converts mouse coordinates from screen space to Penrose Space</p>
     *
     * @param x The mouse x coordinate
     * @param y The mouse y coordinate
     */
    public Point2D.Double screenToPenrose(int x, int y) {

	Point2D.Double currentPoint = new Point2D.Double(x, y);

	return screenToPenrose(currentPoint, viewportTransform);
	
    }


    /**
     * <p>Converts mouse coordinates from screen space to Penrose Space</p>
     *
     * @param p The point to transform
     * @param t The Transformation to apply to map the point
     */
    public static Point2D.Double screenToPenrose(Point2D.Double p,
						 AffineTransform t) {
	// Translate that point into native coordinates

	try {
	    Point2D.Double mapped = (Point2D.Double)t.inverseTransform(p, null);
 	    return mapped;
	}
	catch (NoninvertibleTransformException nie) {
	    // If we can't invert to get the coordinates, set to zero
	    return new Point2D.Double(0.0, 0.0);
	}
    }


    //////////////////
    // Event Handlers
    //////////////////

    /**
     * <p>Handles the event of the user pressing down the mouse button.</p>
     *
     * @param e The Mouse event that was fired
     */
    public void mousePressed(MouseEvent e) {

	pressX = e.getX();
	pressY = e.getY();

	press = screenToPenrose(pressX, pressY);
	
	//PenroseApplet.statusBar.setText("Pressed in " + press.getX() + ", " + press.getY());
	
	// switch!
    }
    

    /**
     * <p>Handles the event of a user dragging the mouse.</p>
     *
     * @param e The Mouse event that was fired
     */
    public void mouseDragged(MouseEvent e) {
	
	//PenroseApplet.statusBar.setText("Draged to " + e.getX() + ", " + e.getY());

	// get the change in mouse coodinates
	releaseX = e.getX();
	releaseY = e.getY();

	// Move the preview based on where the user has dragged
	previewTransform.translate( ( (releaseX - pressX) / previewTransform.getScaleX()),
				    ( (releaseY - pressY) / previewTransform.getScaleY()));

	pressX = releaseX;
	pressY = releaseY;

	// switch!

	// update the screen
	repaint();

    }
    
 
    /**
     * <p>Handles the event of a user releasing the mouse button.</p>
     *
     * @param e The Mouse event that was fired
     */
    public void mouseReleased(MouseEvent e) {

	// get the change in mouse coodinates
	releaseX = e.getX();
	releaseY = e.getY();

	release = screenToPenrose(releaseX, releaseY);
	
	// Move the preview based on where the user has dragged
	previewTransform.translate( ( (releaseX - pressX) / previewTransform.getScaleX()),
				    ( (releaseY - pressY) / previewTransform.getScaleY()));

	pressX = releaseX;
	pressY = releaseY;

	//PenroseApplet.statusBar.setText("Released at " + release.getX() + ", " + release.getY());

	// update the center based on the dragging coordinates
	double xDiff = press.getX() - release.getX();
	double yDiff = press.getY() - release.getY();

	// switch!
	center = new Point2D.Double( (center.getX() + xDiff), 
				     (center.getY() + yDiff));

	// redo the matrix so the coordinates don't get wonky
	recalc();
	repaint();
	
    }
 
   
    /**
     * <p>Handles the event of the user moving the mouse.</p>
     *
     * @param e The Mouse event that was fired
     */
    public void mouseMoved(MouseEvent e) {
	
	Point2D mapped = screenToPenrose(e.getX(), e.getY());
	
	PenroseApplet.statusBar.setText("Penrose Coordinates: " +
					mapped.getX() + ", " + 
					mapped.getY());
	
    }
    

    /**
     * <p>Handles the event of the user clicking the mouse.</p>
     *
     * @param e The Mouse event that was fired
     */
    public void mouseClicked(MouseEvent e) {

	pressX = e.getX();
	pressY = e.getY();
	
	center = screenToPenrose(pressX, pressY);
	    
	// recenter the preview
	previewTransform.translate( ( ((appWidth / 2) - pressX) / previewTransform.getScaleX()) ,
				    ( ((appHeight / 2) - pressY) / previewTransform.getScaleY()) );

	switch (PenroseApplet.currentMode) {
	case PenroseApplet.M_ZOOM_IN:
	    setZoom(2.0);
	    zoom();
	    break;
	    
	case PenroseApplet.M_ZOOM_OUT:
	    setZoom(0.5);
	    zoom();
	    break;
	    
	case PenroseApplet.M_ZOOM:
	    zoom();
	    break;
	    
	default:
	PenroseApplet.statusBar.setText("Recentered At Penrose Coordinates: " +
					center.getX() + ", " + 
					center.getY());
	}

	// reset the coordinates based on the new center
	recalc();

    }
    

    /**
     * <p>Handles the event of the mouse entering this component.</p>
     *
     * @param e The Mouse event that was fired
     */
    public void mouseExited(MouseEvent e) {
	// restore the tool status
	PenroseApplet.statusBar.setText(PenroseApplet.modes[PenroseApplet.currentMode]);
    }


    /**
     * <p>Handles the event of the mouse exiting this component.</p>
     *
     * @param e The Mouse event that was fired
     */
    public void mouseEntered(MouseEvent e) {
	// nothing to do
    }
    
}
