// $Header: /opt/cvs/java/net/logn/penrose/PenroseTiling.java,v 1.47 2001/02/27 04:17:00 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.penrose;

// AWT Components
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.Stroke;
import java.awt.BasicStroke;
import java.awt.RenderingHints;
import java.awt.AlphaComposite;
// Geometry classes
import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Rectangle2D;
import java.awt.Shape;
// Printing support
import java.awt.print.PageFormat;
import java.awt.print.PrinterJob;
import java.awt.print.Printable;
// Sets of objects
import java.util.SortedSet;
// Vectors to store lists of objects
import java.util.Vector;
// And iterators to move over those objects
import java.util.Iterator;
import java.util.Collection;
// Properties support
import java.util.Properties;
import java.util.StringTokenizer;
// File reading (for exporting pictures)
import java.io.File;
import java.io.IOException;
import java.io.FileInputStream;
import java.io.ObjectInputStream;
import java.io.FileOutputStream;
import java.io.ObjectOutputStream;
// Vector exporting classes
import net.logn.util.vectoroutput.*;


/**
 * <p>
 * <b>PenroseTiling</b> provides an interface to all the methods needed to
 * compute the empire of a set of Penrose Tiles.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.47 $
 *
 * Last Modified $Date: 2001/02/27 04:17:00 $ by $Author: jhealy $
 */
public class PenroseTiling implements Printable {


    ////////////
    // Constants
    ////////////

    /** The double value that is closest to the Golden Mean, Tau */
    protected static final double TAU = ((1.0 + (Math.sqrt(5.0))) / 2 );
    
    /** Tolerance within which we match points (used to compensate for
	rounding error */
    // This needs to be at most 0.001 to prevent false matches
    public static final double EPSILON = 0.0000000001;
        
    /** Forces an interval to be longer */
    public static final boolean LONGER = true;

    /** Forces an interval to be shorter */
    public static final boolean SHORTER = false;
    

    // Minnick's Constants
    // Minnick defines several standard vector lengths that are helpful
    // when finding distances between lines in Penrose tiles.  They are
    // defined here.
    
    /** Minnick's "A" Length -- sin(54)((2t + 1)/(2t)) = 1.059017 */
    public static final double MINNICK_A = ( sin(54) * (1 + cos(72)) );
    
    /** Minnick's "B" Length -- 0.25 */
    public static final double MINNICK_B = 0.25;
    
    /** Minnick's "C" Length -- cos(36) + 0.75 = 1.559017 */
    public static final double MINNICK_C = ( cos(36) + 0.75 );
    
    /** Minnick's "D" Length -- a - 1 = 0.059017 */
    public static final double MINNICK_D = ( MINNICK_A - 1 );
    
    /** Minnick's "E" Length -- t - b = 1.368034 */
    public static final double MINNICK_E = ( TAU - MINNICK_B );
    
    /** Minnick's "V" Length -- 1/4(3 - tan(36)/tan(18)) = 0.190983 */
    public static final double MINNICK_V = ( 0.25 * (3 - ( tan(36) / tan(18) ) ) );
    
    /** Minnick's "W" Length -- 0.75 */
    public static final double MINNICK_W = 0.75;
    
    /** Minnick's "X" Length -- (2t + 1)/2t = 1.3090169 = 1 + cos(72) */
    public static final double MINNICK_X = ( 1 + cos(72) );
    
    /** Minnick's "Y" Length -- 1/2t = 0.3090169 = cos(72) */
    public static final double MINNICK_Y = cos(72);
    
    /** Minnick's "Z" Length -- 0.25 */
    public static final double MINNICK_Z = 0.25;
    
    /** Scale factor based on Minnick's Scale for her paper, where
	the SHORT interval is defined as the vectors a+w */
    public static final double SCALE = MINNICK_A + MINNICK_W;
    
    /* The real-coordinate length of a short interval */
    public static final double SHORT = SCALE;

    /* The real-coordinate length of a long interval */
    public static final double LONG = TAU * SCALE;
    
    /* The initial x coordinate of the left of the area to force */
    public static final double INITIAL_LEFT = -7.0;
    
    /* The initial y coordinate of the top of the area to force */
    public static final double INITIAL_TOP = -7.0;
    
    /* The initial x coordinate of the right of the area to force */
    public static final double INITIAL_RIGHT = 7.0;
    
    /* The initial y coordinate of the bottom of the area to force */
    public static final double INITIAL_BOTTOM = 7.0;
    
    /** Minimum amount of overlap between virtual boxes */
    // Use the key distance for a dart
    public static final double BOX_OVERLAP = 2.126627021;

    /** Dimension of each virtual box */
    public static final double BOX_DIM = 10.0;

    /** Origin of the box grid coordinate system, relative to the standard
	coordinate system */
    public static final double BOX_ORIGIN = -(BOX_DIM / 2);
    
    /** Minimum amount of overlap between iterations */
    // Use the maximum pattern width: top of dart
    public static final double ITERATE_OVERLAP = 3.44;

    /** Dimension of each iteration */
    public static final double ITERATE_DIM = 100.0;

    /** Constant for constructor to build a Ace configuration */
    public static final int ACE = 0;

    /** Constant for constructor to build a Deuce configuration */
    public static final int DEUCE = 1;

    /** Constant for constructor to build a Sun configuration */
    public static final int SUN = 2;

    /** Constant for constructor to build a Star configuration */
    public static final int STAR = 3;

    /** Constant for constructor to build a Jack configuration */
    public static final int JACK = 4;

    /** Constant for constructor to build a Queen configuration */
    public static final int QUEEN = 5;

    /** Constant for constructor to build a King configuration */
    public static final int KING = 6;


    ////////////////////
    // Penrose Variables
    ////////////////////

    /** Current Width of the section of the plane we are considering */
    protected static double width = INITIAL_RIGHT - INITIAL_LEFT;

    /** Current Height of the section of the plane we are considering */
    protected static double height = INITIAL_BOTTOM - INITIAL_TOP;

    /** Current X Offset of the section of the plane we are considering */
    protected static double xOffset = 0.0;

    /** Current Y Offset of the section of the plane we are considering */
    protected static double yOffset = 0.0;

    /** Keeps track of whether we've searched for patterns yet or not */
    // if we haven't, then it searches for patterns at the center first.
    private boolean searched = false;

    /** Global Debug Value */
    public static int DEBUG = 0;

    /** Page setup to use when printing */
    protected PageFormat pageSetup = null;

    /** Array of Musical Sequences */
    protected FiveFold fiveFold;

    /** Data structure holding all the sequence axes */
    protected Line2D[] axes = new Line2D[0];

    /** Data structure holding all the forced bars */
    protected Line2D[] forcedBars = new Line2D[0];

    /** Data structure holding all the unforced bars */
    protected Line2D[] unforcedBars = new Line2D[0];

    /** Data structure holding all the intersections */
    protected Ellipse2D[] intersections = new Ellipse2D[0];

    /** Data structure holding all the Kite tiles */
    protected Kite[] kiteTiles = new Kite[0];

    /** Data structure holding all the Dart tiles */
    protected Dart[] dartTiles = new Dart[0];

    /** Number of tiles rendered (used for statistics) */
    protected int numTiles = 0;

    /** Amount of time taken to render (used for statistics) */
    protected long time;

    ////////////////////
    // Drawing Variables
    ////////////////////

    /** Whether to draw axes or not */
    protected boolean drawAxes = false;

    /** Whether to draw forced bars or not */
    protected boolean drawForced = true;

    /** Whether to draw unforced bars or not */
    protected boolean drawUnforced = false;

    /** Whether to draw intersections or not */
    protected boolean drawIntersections = true;

    /** Whether to draw Kite tiles or not */
    protected boolean drawKites = true;

    /** Whether to draw Dart tiles or not */
    protected boolean drawDarts = true;

    /** Whether to fill tiles or not */
    protected boolean fillTiles = true;

    /** Whether to draw tiles or rhombi */
    protected static boolean tilesNotRhombs = true;

    /** Whether to print in color or not */
    protected boolean exportColor = true;

    /** Whether to mark the center or not */
    protected boolean markCenter = true;

    /** Color for background */
    protected static Color backgroundColor = Color.white;

    /** Color for axes */
    protected Color axesColor = Color.yellow;

    /** Color for forced bars */
    protected Color forcedColor = Color.black;

    /** Color for unforced bars */
    protected Color unforcedColor = Color.orange;

    /** Color for intersections */
    protected Color intersectionsColor = Color.red;

    /** Color for Kite (thin rhomb) tiles */
    protected Color kitesColor = Color.blue;

    /** Color for Dart (fat rhomb) tiles */
    protected Color dartsColor = Color.green;

    /** Transparency to use when drawing tiles */
    protected double fillTransparency = 0.2;

    /** Default pen stroke to use when drawing */
    public static final BasicStroke STROKE_DEFAULT = 
	new BasicStroke(0.005f, BasicStroke.CAP_SQUARE,
			BasicStroke.JOIN_MITER, 5.0f);

    /** Compositing rules for screen drawing mode */
    public static final AlphaComposite ALPHA_SCREEN = 
	AlphaComposite.getInstance(AlphaComposite.SRC, 1.0f);

    /** Compositing rules for default printing mode */
    public static final AlphaComposite ALPHA_PRINT = 
	AlphaComposite.getInstance(AlphaComposite.SRC, 1.0f);

    /** Compositing rules for overlay drawing mode */
    protected AlphaComposite ALPHA_OVERLAY = 
	AlphaComposite.getInstance(AlphaComposite.SRC_OVER, (float)fillTransparency);

    /** Compositing rules for default drawing mode */
    public static final AlphaComposite ALPHA_DEFAULT = ALPHA_SCREEN;

    /** Default rendering mode to use when drawing to the screen */
    public static RenderingHints RENDER_SCREEN;

    /** Default rendering mode to use when printing */
    public static RenderingHints RENDER_PRINT;

    /** Default rendering mode to use by default */
    public static RenderingHints RENDER_DEFAULT;

    static {
	// create the rendering hints
	RENDER_SCREEN = new RenderingHints(RenderingHints.KEY_ANTIALIASING,
					   RenderingHints.VALUE_ANTIALIAS_ON);
	
	RENDER_SCREEN.put(RenderingHints.KEY_INTERPOLATION,
			  RenderingHints.VALUE_INTERPOLATION_BICUBIC);

	RENDER_SCREEN.put(RenderingHints.KEY_RENDERING,
			  RenderingHints.VALUE_RENDER_QUALITY);

	RENDER_SCREEN.put(RenderingHints.KEY_COLOR_RENDERING,
			  RenderingHints.VALUE_COLOR_RENDER_QUALITY);

	RENDER_PRINT = new RenderingHints(RenderingHints.KEY_ANTIALIASING,
					  RenderingHints.VALUE_ANTIALIAS_OFF);
	
	RENDER_PRINT.put(RenderingHints.KEY_INTERPOLATION,
			 RenderingHints.VALUE_INTERPOLATION_BICUBIC);

	RENDER_PRINT.put(RenderingHints.KEY_RENDERING,
			 RenderingHints.VALUE_RENDER_QUALITY);

	RENDER_PRINT.put(RenderingHints.KEY_COLOR_RENDERING,
			 RenderingHints.VALUE_COLOR_RENDER_QUALITY);

	RENDER_DEFAULT = RENDER_SCREEN;
    }

    /**
     * <p>
     * Constructs a Penrose Tiling with the Sun configuration at its center.
     * </p>
     */
    public PenroseTiling() {

	// The Sun configuration does not force any additional bars,
	// so it is the most harmless set to use as a default
	setInitialConfiguration(SUN);

    }


    /**
     * <p>
     * Constructs a PenroseTiling with the specified Vertex configuration
     * at its center.
     * </p>
     *
     * @param vertexConfig The vertex configuration to use at the center
     * of the tiling.
     */
    public PenroseTiling(int vertexConfig) {
	
	this();
	
	setInitialConfiguration(vertexConfig);

    }


    /**
     * <p>
     * Saves the current Ammann Bar configuration to the file provided.
     * </p>
     *
     * @param file The file to save the configuration to
     */
    public void saveConfiguration(File file) throws IOException {
	FileOutputStream outStream = new FileOutputStream(file);
	ObjectOutputStream output = new ObjectOutputStream(outStream);

        output.writeObject(fiveFold);

        output.flush();
        outStream.close();

    }


    /**
     * <p>
     * Loads an Ammann Bar configuration from the file provided.
     * </p>
     *
     * @param file The file to load the configuration from
     */
    public void loadConfiguration(File file) throws IOException {

	FiveFold temp;

	FileInputStream inStream = new FileInputStream(file);
	ObjectInputStream input = new ObjectInputStream(inStream);

	try {
	    temp = (FiveFold)input.readObject();
	}
	catch (ClassNotFoundException cnfe) {
	    temp = null;
	}

	inStream.close();

	setInitialConfiguration(temp);
    }


    /**
     * <p>
     * Erases the current configuration (if any) and sets it to the vertex
     * configuration defined by the constant passed to the method.
     * </p>
     *
     * @param vertexConfig The vertex configuration to use at the center
     * of the tiling
     */
    public void setInitialConfiguration(int vertexConfig) {

	FiveFold newFiveFold = null;

	switch (vertexConfig) {
	    
	case ACE:
	    newFiveFold = aceConfiguration();
	    break;
	    
	case DEUCE:
	    newFiveFold = deuceConfiguration();
	    break;
	    
	case SUN:
	    newFiveFold = sunConfiguration();
	    break;
	    
	case STAR:
	    newFiveFold = starConfiguration();
	    break;
	    
	case JACK:
	    newFiveFold = jackConfiguration();
	    break;
	    
	case QUEEN:
	    newFiveFold = queenConfiguration();
	    break;
	    
	case KING:
	    newFiveFold = kingConfiguration();
	    break;
	    
	}
	
	if (newFiveFold != null) {
	    // now set the new configuration
	    setInitialConfiguration(newFiveFold);
	}
    }


    /**
     * <p>
     * Erases the current configuration (if any) and sets it to the vertex
     * configuration provided.
     * </p>
     *
     * @param newFiveFold The new musical sequence interaction to use.
     */
    public void setInitialConfiguration(FiveFold newFiveFold) {

	fiveFold = newFiveFold;

	// clear all existing drawing info and load the new bars in

	intersections = new Ellipse2D[0];
	kiteTiles = new Kite[0];
	dartTiles = new Dart[0];

	double[] bbox = getBoundingBox();

	axes = fiveFold.getAxes(bbox[0], bbox[1], bbox[2], bbox[3]);
	forcedBars = fiveFold.getForcedLines(bbox[0], bbox[1],
					     bbox[2], bbox[3]);
	unforcedBars = fiveFold.getUnforcedLines(bbox[0], bbox[1],
						 bbox[2], bbox[3]);

	searched = false;
    }


    /**
     * <p>
     * Erases the current configuration (if any) and sets it to the
     * arrangements of bars defined by the parameters.
     * </p>
     *
     * @param barOffsets An array containing the offsets from (0, 0) for
     * each of the Ammann bar sequences as a two-double array
     * @param forcedBars An array containing all the forced bars that should
     * be in the initial configuration
     * @param forcedLengths An array containing the lengths of those forced
     * bars
     */
    public void getInitialConfiguration(double[][] barOffsets,
					long[][] forcedBars,
					boolean[][] forcedLengths) {

	if ( (barOffsets.length != FiveFold.NUMSEQUENCES) ||
	     (forcedBars.length != FiveFold.NUMSEQUENCES) ||
	     (forcedLengths.length != FiveFold.NUMSEQUENCES) ) {
	    
	    System.err.println("Error!  Cannot set initial configuration." +
			       "\nIncorrect number of Musical Sequence " +
			       "\nconfigurations provided");
	}
	else {

	    fiveFold = new FiveFold();
	    
	    for (int i = 0; i < FiveFold.NUMSEQUENCES; i++) {
		
		fiveFold.sequences[i].centerX = barOffsets[i][0];
		fiveFold.sequences[i].centerY = barOffsets[i][1];

		for (int j = 0; j < forcedBars[i].length; j++) {

		    fiveFold.sequences[i].force(forcedBars[i][j],
						forcedLengths[i][j]);
		}

	    }

	}
    }


    /**
     * <p>
     * Returns the bar offsets for the Ace configuration.
     * </p>
     *
     * @return FiveFold The interaction of Musical Sequences that produces
     * the Ace configuration
     */
    public static FiveFold aceConfiguration() {
	
	FiveFold plane = new FiveFold();

	// 0 Degree Axis
	plane.sequences[0].setZeroBar(MINNICK_A);

	// 72 Degree Axis
	plane.sequences[1].setZeroBar(MINNICK_A);

	// 144 Degree Axis
	plane.sequences[2].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	// 216 Degree Axis
	plane.sequences[3].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	// 288 Degree Axis
	plane.sequences[4].setZeroBar(MINNICK_A);

	// All done

	return plane;
    }


    /**
     * <p>
     * Returns the bar offsets for the Deuce configuration.
     * </p>
     *
     * @return FiveFold The interaction of Musical Sequences that produces
     * the Deuce configuration
     */
    public static FiveFold deuceConfiguration() {
	
	FiveFold plane = new FiveFold();

	// 0 Degree Axis
	plane.sequences[0].setZeroBar(MINNICK_A);

	// 72 Degree Axis
	plane.sequences[1].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	// 144 Degree Axis
	plane.sequences[2].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	// 216 Degree Axis
	plane.sequences[3].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	// 288 Degree Axis
	plane.sequences[4].setZeroBar(MINNICK_A);

	// All done

	return plane;
    }


    /**
     * <p>
     * Returns the bar offsets for the Sun configuration.
     * </p>
     *
     * @return FiveFold The interaction of Musical Sequences that produces
     * the Sun configuration
     */
    public static FiveFold sunConfiguration() {
	
	FiveFold plane = new FiveFold();

	// 0 Degree Axis
	plane.sequences[0].setZeroBar(MINNICK_B);

	// 72 Degree Axis
	plane.sequences[1].setZeroBar(MINNICK_B);

	// 144 Degree Axis
	plane.sequences[2].setZeroBar(MINNICK_B);

	// 216 Degree Axis
	plane.sequences[3].setZeroBar(MINNICK_B);

	// 288 Degree Axis
	plane.sequences[4].setZeroBar(MINNICK_B);

	// All done

	return plane;
    }


    /**
     * <p>
     * Returns the bar offsets for the Star configuration.
     * </p>
     *
     * @return FiveFold The interaction of Musical Sequences that produces
     * the Star configuration
     */
    public static FiveFold starConfiguration() {
	
	FiveFold plane = new FiveFold();

	// 0 Degree Axis
	plane.sequences[0].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	plane.sequences[0].force(1, LONGER);

	// 72 Degree Axis
	plane.sequences[1].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	plane.sequences[1].force(1, LONGER);

	// 144 Degree Axis
	plane.sequences[2].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	plane.sequences[2].force(1, LONGER);

	// 216 Degree Axis
	plane.sequences[3].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	plane.sequences[3].force(1, LONGER);

	// 288 Degree Axis
	plane.sequences[4].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	plane.sequences[4].force(1, LONGER);

	// All done

	return plane;
    }


    /**
     * <p>
     * Returns the bar offsets for the Jack configuration.
     * </p>
     *
     * @return FiveFold The interaction of Musical Sequences that produces
     * the Jack configuration
     */
    public static FiveFold jackConfiguration() {
	
	FiveFold plane = new FiveFold();

	// 0 Degree Axis
	plane.sequences[0].setZeroBar(MINNICK_E);
	// The only tile that can sit on top of these two kites is a
	// dart.  Therefore, we can force the bar for the dart here.
	plane.sequences[0].force(-1, LONGER);

	// 72 Degree Axis
	plane.sequences[1].setZeroBar(MINNICK_Z);
	// The only tile that can sit on top of these two kites is a
	// dart.  Therefore, we can force the bar for the dart here.
	plane.sequences[1].force(-1, LONGER);

	// 144 Degree Axis
	plane.sequences[2].setZeroBar(MINNICK_B);

	// 216 Degree Axis
	plane.sequences[3].setZeroBar(MINNICK_B);

	// 288 Degree Axis
	plane.sequences[4].setZeroBar(MINNICK_Z);
	// The only tile that can sit on top of these two kites is a
	// dart.  Therefore, we can force the bar for the dart here.
	plane.sequences[4].force(-1, LONGER);

	// All done

	return plane;
    }


    /**
     * <p>
     * Returns the bar offsets for the Queen configuration.
     * </p>
     *
     * @return FiveFold The interaction of Musical Sequences that produces
     * the Queen configuration
     */
    public static FiveFold queenConfiguration() {
	
	FiveFold plane = new FiveFold();

	// 0 Degree Axis
	plane.sequences[0].setZeroBar(MINNICK_A);

	// 72 Degree Axis
	plane.sequences[1].setZeroBar(-MINNICK_W);

	plane.sequences[1].force(1, SHORTER);

	// 144 Degree Axis
	plane.sequences[2].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	plane.sequences[2].force(1, LONGER);

	// 216 Degree Axis
	plane.sequences[3].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	plane.sequences[3].force(1, LONGER);

	// 288 Degree Axis
	plane.sequences[4].setZeroBar(-MINNICK_W);

	plane.sequences[4].force(1, SHORTER);

	// All done

	return plane;
    }


    /**
     * <p>
     * Returns the bar offsets for the King configuration.
     * </p>
     *
     * @return FiveFold The interaction of Musical Sequences that produces
     * the King configuration
     */
    public static FiveFold kingConfiguration() {
	
	FiveFold plane = new FiveFold();

	// 0 Degree Axis
	plane.sequences[0].setZeroBar(-MINNICK_W);

	plane.sequences[0].force(1, SHORTER);

	// 72 Degree Axis
	plane.sequences[1].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	plane.sequences[1].force(1, LONGER);

	// 144 Degree Axis
	plane.sequences[2].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	plane.sequences[2].force(1, LONGER);

	// 216 Degree Axis
	plane.sequences[3].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	plane.sequences[3].force(1, LONGER);

	// 288 Degree Axis
	plane.sequences[4].setZeroBar(-(MINNICK_X + MINNICK_Y + MINNICK_Z));

	plane.sequences[4].force(1, LONGER);

	// All done

	return plane;
    }


    /**
     * <p>
     * Sets the current Width and Height of the area under consideration.
     * </p>
     *
     * @param newWidth The new width to consider
     * @param newHeight The new height to consider
     */
    public void setDimensions(double newWidth, double newHeight) {

	width = newWidth;
	height = newHeight;

    }


    /**
     * <p>
     * Sets the offset from the origin for the center of the area
     * of the tiling to view.
     * </p>
     *
     * @param x The new x offset
     * @param y The new y offset
     */
    public void setOffset(double x, double y) {

	xOffset = x;
	yOffset = y;

    }


    /**
     * <p>
     * Sets the offset from the origin for the center of the area
     * of the tiling to view.
     * </p>
     *
     * @param transform A transform mapping the center to its new position
     */
    public void setOffset(AffineTransform transform) {

	setOffset(transform.getTranslateX(), transform.getTranslateY());

    }


    /**
     * <p>
     * Recompute all of the tile forcings.
     * </p>
     */
    public void recomputeEmpire() {
	
	recomputeEmpire(null);

    }


    /**
     * <p>
     * Recompute all of the tile forcings.  Provides graphical updates
     * to the canvas provided.
     * </p>
     *
     * @param canvas The graphics component to render the progress to
     */
    public void recomputeEmpire(Graphics2D canvas) {


	if (!searched) {
	    // If we have never computed anything on this empire, then
	    // compute the center of the empire now.  This is where most of
	    // the forcings are likely to occur, so it makes sense to force
	    // here first, and then zoom out and check the rest of the plane.
	    recomputeArea(INITIAL_LEFT, INITIAL_TOP,
			  INITIAL_RIGHT, INITIAL_BOTTOM,
			  null);

	    searched = true;
	}

	// now compute the area for the whole plane, and check that
	
	// left side
        double left = xOffset - (width / 2) - BOX_OVERLAP;
	// upper side
        double top = yOffset - (height / 2) - BOX_OVERLAP;
	// right side
        double right = left + width + (2 * BOX_OVERLAP);
	// bottom side
        double bottom = top + height + (2 * BOX_OVERLAP);

	// clear the canvas
	if (canvas != null) {
	    
	    Rectangle2D clearRect = new Rectangle2D.Double(left, top,
							   (right - left),
							   (bottom - top));
	    // set to opaque
	    canvas.setComposite(ALPHA_SCREEN);
	    
	    canvas.setColor(backgroundColor);
	    canvas.fill(clearRect);
	}

	recomputeArea(left, top, right, bottom, canvas);

	// Now that everything has been forced, get the bars

	axes = fiveFold.getAxes(left, top, right, bottom);
	
	forcedBars = fiveFold.getForcedLines(left, top, right, bottom);
	
	unforcedBars = fiveFold.getUnforcedLines(left, top, right, bottom);

	// all done!
    }


    /**
     * <p>
     * Recompute all of the tile forcings for the given area.  Provides 
     * graphical updates to the canvas provided.
     * </p>
     *
     * @param lSide The x value of the left side of the area to compute
     * @param uSide The y value of the top side of the area to compute
     * @param rSide The x value of the right side of the area to compute
     * @param bSide The y value of the bottom side of the area to compute
     * @param canvas The graphics component to render the progress to
     */
    public void recomputeArea(double lSide, double uSide,
			      double rSide, double bSide, Graphics2D canvas) {

	if (canvas != null) {
	    // set the rendering hints
	    canvas.setRenderingHints(RENDER_DEFAULT);
	    // set the stroke width
	    canvas.setStroke(STROKE_DEFAULT);
	    // set the transparency
	    canvas.setComposite(ALPHA_OVERLAY);
	}

	// keep track of whether we've forced or not
	boolean newForcings = true;

	// Collections to hold all the tiles we find
	Collection kiteMatches = new Vector(0);
	Collection dartMatches = new Vector(0);
	Collection doublekiteMatches = new Vector(0);

	// collections to hold the points that we find
	SortedSet points = null;
	Vector boundaries = null;

	while (newForcings) {
	    
	    newForcings = false;

	    // get the intersections
	    points = fiveFold.findIntersectionPoints(lSide, uSide,
						     rSide, bSide);

	    // prep the array to hold the intersection point dots
	    intersections = new Ellipse2D[points.size()];
	    int intersectionsIndex = 0;
	    
	    Iterator pointIterator = points.iterator();
	    // Collection to hold points that lie on a boundary of a box
	    boundaries = new Vector();
	    double layer = -1.0;
	    double theta = -1.0;
	    
	    if (canvas != null) {
		canvas.setComposite(ALPHA_OVERLAY);
		
		Rectangle2D box = null;

		while (pointIterator.hasNext()) {
		    
		    IntersectionPoint point = (IntersectionPoint)pointIterator.next();
		    if ( (point.boxLayer != layer) || (point.boxTheta != theta) ) {
			
			layer = point.boxLayer;
			theta = point.boxTheta;
			
			// Remeber this point, as it is on the boundary of a
			// virtual box
			boundaries.add(point);
			
			box = null;

		    }

		    if (box == null) {
			canvas.setColor(Color.cyan);
			box = point.getBoxRect();
			if (box != null) {
			    canvas.fill(box);
			}
		    }
		    
		    Shape dot = point.getShape();
		    intersections[intersectionsIndex] = (Ellipse2D)dot;
		    intersectionsIndex++;
		    
		    canvas.setColor(intersectionsColor);
		    canvas.fill(dot);
		}
	    }
	    else {
		while (pointIterator.hasNext()) {
		    
		    IntersectionPoint point = (IntersectionPoint)pointIterator.next();
		    
		    if ( (point.boxLayer != layer) || (point.boxTheta != theta) ) {
			
			layer = point.boxLayer;
			theta = point.boxTheta;
			
			// Remeber this point, as it is on the boundary of a
			// virtual box
			boundaries.add(point);
					    
			intersections[intersectionsIndex] = (Ellipse2D)point.getShape();
			intersectionsIndex++;
		    
		    }
		}
	    }

	    // find constellations
	    if (canvas != null) {
		canvas.setColor(dartsColor);
	    }

	    dartMatches = Dart.getConstellations(points, fiveFold,
						 boundaries, canvas);

	    doublekiteMatches = DoubleKite.getConstellations(points, fiveFold,
							     boundaries, canvas);

	    // Don't force kites (nothing to do there)

	    // Darts
	    newForcings = newForcings || forceNew(dartMatches);

	    // Double-Kites
	    newForcings = newForcings || forceNew(doublekiteMatches);

	}

	// kites never force new information, so we can look for them after
	// we've finished forcing everything
	if (canvas != null) {
	    canvas.setColor(kitesColor);
	}

	kiteMatches = Kite.getConstellations(points, fiveFold,
					     boundaries, canvas);
	
	// dump the kites out to the arrays
	kiteTiles = new Kite[kiteMatches.size()];
	int kiteIndex = 0;
	
	for (Iterator kiteIterator = kiteMatches.iterator();
	     kiteIterator.hasNext();
	     ) {
	    
	    kiteTiles[kiteIndex] = (Kite)kiteIterator.next();
	    kiteIndex++;
	    
	}
	
	// dump the darts out to the arrays
	dartTiles = new Dart[dartMatches.size()];
	int dartIndex = 0;
	
	for (Iterator dartIterator = dartMatches.iterator();
	     dartIterator.hasNext();
	     ) {
	    
	    dartTiles[dartIndex] = (Dart)dartIterator.next();
	    dartIndex++;
	    
	}
	

    }


    /**
     * <p>
     * Attempts to force new bars using the constellations provided
     * </p>
     *
     * @param constellations The collection of Constellations to convert
     *
     * @return boolean True, if any new forcings occured
     */
    public boolean forceNew(Collection constellations) {
	
	boolean forced = false;
	
	for (Iterator tiles = constellations.iterator();
	     tiles.hasNext();
	     ) {
	    
	    Constellation tile = (Constellation)tiles.next();
	    
	    forced = forced || tile.forceBars(fiveFold);
	    
	}

	return forced;
    }


    /**
     * <p>
     * Renders the tiling to a graphics object, using the default drawing
     * properties.
     * </p>
     *
     * @param canvas The Graphics object to render to
     */
    public void renderTiling(Graphics2D canvas) {

	renderTiling(canvas, STROKE_DEFAULT, ALPHA_DEFAULT, RENDER_DEFAULT);

    }


    /**
     * <p>
     * Renders the tiling to a graphics object using the drawing properties
     * specified.
     * </p>
     *
     * @param canvas The Graphics object to render to
     * @param stroke The default pen stroke to use
     * @param compositing The default alpha compositing to use
     * @param renderHints The default rendering hints to use
     */
    public void renderTiling(Graphics2D canvas,
			     Stroke stroke,
			     AlphaComposite compositing,
			     RenderingHints renderHints) {

	canvas.setRenderingHints(renderHints);

 	canvas.setStroke(stroke);

	canvas.setComposite(compositing);

	if (drawAxes) {
	    canvas.setColor(axesColor);
	    for (int i = 0; i < axes.length; i++) {
		canvas.draw(axes[i]);
	    }
	}

	if (drawForced) {
	    canvas.setColor(forcedColor);
	    for (int i = 0; i < forcedBars.length; i++) {
		canvas.draw(forcedBars[i]);
	    }
	}

	if (drawUnforced) {
	    canvas.setColor(unforcedColor);
	    for (int i = 0; i < unforcedBars.length; i++) {
		canvas.draw(unforcedBars[i]);
	    }
	}

	if (drawIntersections) {
	    canvas.setColor(intersectionsColor);
	    for (int i = 0; i < intersections.length; i++) {
		canvas.draw(intersections[i]);
	    }
	}

	if (drawKites) {
	    canvas.setColor(kitesColor);
	    for (int i = 0; i < kiteTiles.length; i++) {
		canvas.draw(kiteTiles[i].getShape());
	    }
	    if (fillTiles) {
		canvas.setComposite(ALPHA_OVERLAY);
		for (int j = 0; j < kiteTiles.length; j++) {
		    canvas.fill(kiteTiles[j].getShape());
		}
		canvas.setComposite(compositing);
	    }	
	}	
	
	if (drawDarts) {
	    canvas.setColor(dartsColor);
	    for (int i = 0; i < dartTiles.length; i++) {
		canvas.draw(dartTiles[i].getShape());
	    }
	    if (fillTiles) {
		canvas.setComposite(ALPHA_OVERLAY);
		for (int j = 0; j < dartTiles.length; j++) {
		    canvas.fill(dartTiles[j].getShape());
		}
		canvas.setComposite(compositing);
	    }	
	}	
	
	// draw a circle at the center
	if (markCenter) {
	    canvas.setComposite(ALPHA_OVERLAY);
	    canvas.setColor(Color.black);
	    canvas.fillOval(-1, -1, 2, 2);
	    canvas.setComposite(compositing);
	}

    }

    
    /**
     * <p>
     * Prints the current rendering of the Penrose Tiling
     * </p>
     * 
     * @return String The error message, or null if there were no problems
     */
    public String print() {

        PrinterJob printJob = PrinterJob.getPrinterJob();

	PageFormat pageSetup = printJob.pageDialog(printJob.defaultPage());
	
	printJob.setPrintable(this, pageSetup);

        if (printJob.printDialog()) {
            try {
                printJob.print();
            } catch (Exception ex) {
                return ex.getMessage();
            }
        }
	
	return null;
    }
    

    /**
     * <p>
     * Prints the current rendering of the Penrose Tiling
     * </p>
     * 
     */
    public int print(java.awt.Graphics graphics,
		     PageFormat pageFormat, int pageIndex) {

	if (pageIndex > 0) {
            return Printable.NO_SUCH_PAGE;
        }

	// We must translate the image so that it is centered on the page

	Graphics2D g2 = (Graphics2D)graphics;

	double pageWidth = pageFormat.getImageableWidth();
	double pageHeight = pageFormat.getImageableHeight();

	double xMargin = (pageFormat.getWidth() - pageWidth) / 2;
	double yMargin = (pageFormat.getHeight() - pageHeight) / 2;

	g2.translate(((pageWidth / 2) + xMargin), ((pageHeight / 2) + yMargin));

	if (DEBUG > 3) {
	    System.out.println("Page dimensions: " + pageWidth +
			       ", " + pageHeight);
	}

	double xScale = pageWidth / width;
	double yScale = pageHeight / height;
	double pageScale;

	if (xScale < yScale) {
	    pageScale = xScale;
	}
	else {
	    pageScale = yScale;
	}

	g2.translate(-(xOffset * pageScale), -(yOffset * pageScale));

	g2.scale(pageScale, pageScale);

	// draw a border around the screen, so we know what the applet
	// thinks its boundaries are

	Rectangle2D border = new Rectangle2D.Double((xOffset - (width / 2) - 1),
						    (yOffset - (height / 2) - 1),
						    (width + 2),
						    (height + 2));

	if (DEBUG > 3) {
	    g2.setColor(Color.black);
	    g2.draw(border);
	}
	else {
	    g2.clip(border);
	}

	if (!exportColor) {
	    // convert colors to black and white
	    Color bg = backgroundColor;
	    Color a = axesColor;
	    Color f = forcedColor;
	    Color uf = unforcedColor;
	    Color i = intersectionsColor;
	    Color k = kitesColor;
	    Color d = dartsColor;

	    backgroundColor = Color.white;
	    axesColor = Color.black;
	    forcedColor = Color.black;
	    unforcedColor = Color.black;
	    intersectionsColor = Color.black;
	    kitesColor = Color.black;
	    dartsColor = Color.black;

	    // now draw the page in black and white
	    renderTiling(g2, STROKE_DEFAULT, ALPHA_PRINT, RENDER_PRINT);

	    // and set the colors back
	    backgroundColor = bg;
	    axesColor = a;
	    forcedColor = f;
	    unforcedColor = uf;
	    intersectionsColor = i;
	    kitesColor = k;	    
	    dartsColor = d;	    
	}
	else {
	    renderTiling(g2, STROKE_DEFAULT, ALPHA_PRINT, RENDER_PRINT);
	}

        return Printable.PAGE_EXISTS;

    }


    /**
     * Exports the current tiling as a PNG.
     *
     * @param outFile The file to write to
     * @param wide The width (in pixels) of the PNG image
     * @param tall The height (in pixels) of the PNG image
     */
    public void exportPNG(File outFile, int wide, int tall) throws IOException {	
	PNG png = getPNGStream(outFile, wide, tall);

	exportVector(png);

    }


    /**
     * Exports the current tiling as EPS.
     *
     * @param outFile The file to write to
     */
    public void exportEPS(File outFile) throws IOException {

	Postscript eps = getPostscriptStream(outFile);
	
	exportVector(eps);

    }


    /**
     * Exports the current tiling as PDF.
     *
     * @param outFile The file to write to
     */
    public void exportPDF(File outFile) throws IOException {

	PDF pdf = getPDFStream(outFile);

	exportVector(pdf);

    }


    /**
     * Gets a PNG Vector Output Object to write to.
     *
     * @param outFile The file to write to
     * @param wide The width (in pixels) of the PNG image
     * @param tall The height (in pixels) of the PNG image
     */
    public PNG getPNGStream(File outFile, int wide, int tall) throws IOException {

	double[] bbox = getBoundingBox();

	// define a background if we're exporting in color
	Color bg = null;
	if (exportColor) {
	    bg = backgroundColor;
	}
	
	PNG png = new PNG(outFile, bg, wide, tall, bbox[0], bbox[1],
			  bbox[2], bbox[3]);

	return png;

    }


    /**
     * Gets a Postscript Vector Output Object to write to.
     *
     * @param outFile The file to write to
     */
    public Postscript getPostscriptStream(File outFile) throws IOException {

	double[] bbox = getBoundingBox();
	
	// define a background if we're exporting in color
	Color bg = null;
	if (exportColor) {
	    bg = backgroundColor;
	}
	
	Postscript eps = new Postscript(outFile, bg,
					bbox[0], bbox[1],
					bbox[2], bbox[3],
					"Penrose Empire Rendering",
					null, null,
					"penrose tiling empire",
					"LogN Penrose Empire Software");
	
	return eps;

    }


    /**
     * Gets a PDF Vector Output Object to write to.
     *
     * @param outFile The file to write to
     */
    public PDF getPDFStream(File outFile) throws IOException {

	double[] bbox = getBoundingBox();
	
	// define a background if we're exporting in color
	Color bg = null;
	if (exportColor) {
	    bg = backgroundColor;
	}
	
	PDF pdf = new PDF(outFile, bg,
			  bbox[0], bbox[1],
			  bbox[2], bbox[3],
			  "Penrose Empire Rendering",
			  null, null,
			  "penrose tiling empire",
			  "LogN Penrose Empire Software");

	return pdf;

    }


    /**
     * Returns the current bounding box for the view we have
     */
    public double[] getBoundingBox() {

	double[] box = new double[4];

	// left side
        box[0] = xOffset - (width / 2);
	// upper side
        box[1] = yOffset - (height / 2);
	// right side
        box[2] = box[0] + width;
	// bottom side
        box[3] = box[1] + height;

	return box;

    }


    /**
     * Exports the current tiling to a vector output.
     *
     * @param output The output object to use
     */
    protected void exportVector(VectorOutput output) throws IOException {

	output.prepare();

	output.setStroke(STROKE_DEFAULT);

	output.setStrokeTransparency(1.0);
	output.setFillTransparency(0.2);
	
	if (drawKites) {
	    Kite templateKite = new Kite();
	    output.createTemplate("kite", templateKite.getShape());

	    AffineTransform[] mappings = new AffineTransform[kiteTiles.length];
	    for (int i = 0; i < mappings.length; i++) {
		mappings[i] = kiteTiles[i].mapping;
	    }

	    if (exportColor) {
		output.setStrokeColor(kitesColor);
		output.setFillColor(kitesColor);
	    }
	    if (fillTiles) {
		output.drawTemplates("kite", mappings);
	    }
	    else {
		output.strokeTemplates("kite", mappings);
	    }
	}	
	
	if (drawDarts) {
	    Dart templateDart = new Dart();
	    output.createTemplate("dart", templateDart.getShape());

	    AffineTransform[] mappings = new AffineTransform[dartTiles.length];
	    for (int i = 0; i < mappings.length; i++) {
		mappings[i] = dartTiles[i].mapping;
	    }

	    if (exportColor) {
		output.setStrokeColor(dartsColor);
		output.setFillColor(dartsColor);
	    }
	    if (fillTiles) {
		output.drawTemplates("dart", mappings);
	    }
	    else {
		output.strokeTemplates("dart", mappings);
	    }
	}
	
	// draw a circle at the center
	if (markCenter) {
	    if (exportColor) {
		output.setStrokeColor(Color.black);
		output.setFillColor(Color.black);
	    }

	    if (fillTiles) {
		output.fillShape(new Ellipse2D.Double(-1, -1, 2, 2));
	    }
	    else {
		output.strokeShape(new Ellipse2D.Double(-1, -1, 2, 2));
	    }
	}
	
	if (drawIntersections) {
	    if (exportColor) {
		output.setStrokeColor(intersectionsColor);
	    }
	    output.strokeShapes(intersections);
	}
	
	if (drawAxes) {
	    if (exportColor) {
		output.setStrokeColor(axesColor);
	    }
	    output.strokeShapes(axes);
	}
	
	if (drawForced) {
	    if (exportColor) {
		output.setStrokeColor(forcedColor);
	    }
	    output.strokeShapes(forcedBars);
	}
	
	if (drawUnforced) {
	    if (exportColor) {
		output.setStrokeColor(unforcedColor);
	    }
	    output.strokeShapes(unforcedBars);
	}
	
	output.finish();
	
    }


    /**
     * <p>
     * Recompute all of the tile forcings and render the final empire to
     * the output provided.
     * </p>
     *
     * @param output The VectorOutput stream to render the Empire to
     */
    public void streamEmpire(VectorOutput output) throws IOException {

	time = System.currentTimeMillis();

	if (!searched) {

	    System.out.println("Computing center of tiling...");

	    // If we have never computed anything on this empire, then
	    // compute the center of the empire now.  This is where most of
	    // the forcings are likely to occur, so it makes sense to force
	    // here first, and then zoom out and check the rest of the plane.
	    recomputeArea(INITIAL_LEFT, INITIAL_TOP,
			  INITIAL_RIGHT, INITIAL_BOTTOM,
			  null);

	    searched = true;
	}

	// now compute the area for the whole plane, and check that
	
	// left side
        double left = xOffset - (width / 2) - BOX_OVERLAP;
	// upper side
        double top = yOffset - (height / 2) - BOX_OVERLAP;
	// right side
        double right = left + width + (2 * BOX_OVERLAP);
	// bottom side
        double bottom = top + height + (2 * BOX_OVERLAP);

	output.prepare();

	// set the stroke width
	output.setStroke(STROKE_DEFAULT);

	output.setStrokeTransparency(1.0);
	output.setFillTransparency(fillTransparency);

	System.out.println("Beginning iteration over the plane");

	int rowsRemaining = (int)Math.ceil((right - left) / 
					   (ITERATE_DIM - ITERATE_OVERLAP));

	int colsRemaining = (int)Math.ceil((bottom - top) / 
				       (ITERATE_DIM - ITERATE_OVERLAP));

	int cols, blocks;
	long unitTime;

	time = System.currentTimeMillis() - time;

	// iterate over the plane to minimize memory requirements
	for (double slideLeft = left;
	     slideLeft < right;
	     slideLeft += (ITERATE_DIM - ITERATE_OVERLAP)) {

	    double slideRight = slideLeft + ITERATE_DIM;

	    cols = colsRemaining;
	    
	    for (double slideBottom = top;
		 slideBottom < bottom;
		 slideBottom += (ITERATE_DIM - ITERATE_OVERLAP)) {
		
		double slideTop = slideBottom + ITERATE_DIM;
		
		unitTime = streamArea(slideLeft, slideBottom,
				      slideRight, slideTop,
				      output);

		cols--;

		time += unitTime;

		blocks = ((rowsRemaining - 1) * colsRemaining) + cols;

		int minutes, hours;

		hours = (int)(unitTime * blocks) / 3600000;
		minutes = (int)((unitTime * blocks) / 60000) - (hours * 60);

		System.out.println(blocks + " blocks remaining " +
				   "(Estimated time remaining: " +
				   hours + " hours, " + minutes + " minutes)");

	    }

	    rowsRemaining--;
	}
	
	// draw a circle at the center
	if (markCenter) {
	    if (exportColor) {
		output.setStrokeColor(Color.black);
		output.setFillColor(Color.black);
	    }

	    if (fillTiles) {
		output.fillShape(new Ellipse2D.Double(-1, -1, 2, 2));
	    }
	    else {
		output.strokeShape(new Ellipse2D.Double(-1, -1, 2, 2));
	    }
	}
	
	if (drawAxes) {
	    if (exportColor) {
		output.setStrokeColor(axesColor);
	    }
	    output.strokeShapes(fiveFold.getAxes(left, top, right, bottom));
	}
	
	if (drawForced) {
	    if (exportColor) {
		output.setStrokeColor(forcedColor);
	    }
	    output.strokeShapes(fiveFold.getForcedLines(left, top, right, bottom));
	}
	
	if (drawUnforced) {
	    if (exportColor) {
		output.setStrokeColor(unforcedColor);
	    }
	    output.strokeShapes(fiveFold.getUnforcedLines(left, top, right, bottom));
	}
	
	output.finish();
	
    }


    /**
     * <p>
     * Recompute all of the tile forcings for the given area.  Provides 
     * graphical updates to the canvas provided.
     * </p>
     *
     * @param lSide The x value of the left side of the area to compute
     * @param uSide The y value of the top side of the area to compute
     * @param rSide The x value of the right side of the area to compute
     * @param bSide The y value of the bottom side of the area to compute
     * @param output The VectorOutput devide to render to
     *
     * @return long The number of milliseconds for this area computation
     */
    public long streamArea(double lSide, double uSide,
			   double rSide, double bSide,
			   VectorOutput output) throws IOException {

	long time = System.currentTimeMillis();

	// keep track of whether we've forced or not
	boolean newForcings = true;

	// Collections to hold all the tiles we find
	Collection kiteMatches = new Vector(0);
	Collection dartMatches = new Vector(0);
	Collection doublekiteMatches = new Vector(0);

	// collections to hold the points that we find
	SortedSet points = null;
	Vector boundaries = null;

	while (newForcings) {
	    
	    newForcings = false;

	    // get the intersections
	    points = fiveFold.findIntersectionPoints(lSide, uSide,
						     rSide, bSide);

	    Iterator pointIterator = points.iterator();
	    // Collection to hold points that lie on a boundary of a box
	    boundaries = new Vector();
	    double layer = -1.0;
	    double theta = -1.0;
	    
	    while (pointIterator.hasNext()) {
		
		IntersectionPoint point = (IntersectionPoint)pointIterator.next();
		if ( (point.boxLayer != layer) || (point.boxTheta != theta) ) {
		    
		    layer = point.boxLayer;
		    theta = point.boxTheta;
		    
		    // Remeber this point, as it is on the boundary of a
		    // virtual box
		    boundaries.add(point);
		    
		}
		
	    }
	    
	    // find constellations

	    dartMatches = Dart.getConstellations(points, fiveFold,
						 boundaries, null);
	    
	    doublekiteMatches = DoubleKite.getConstellations(points, fiveFold,
							     boundaries, null);

	    // Don't force kites (nothing to do there)

	    // Darts
	    newForcings = newForcings || forceNew(dartMatches);

	    // Double-Kites
	    newForcings = newForcings || forceNew(doublekiteMatches);

	}

	// kites never force new information, so we can look for them after
	// we've finished forcing everything

	kiteMatches = Kite.getConstellations(points, fiveFold,
					     boundaries, null);

	Shape shape;

	numTiles += (kiteMatches.size() + dartMatches.size());

	if (fillTiles) {
	    // stroke and fill shapes
	    if (exportColor) {
		output.setStrokeColor(kitesColor);
		output.setFillColor(kitesColor);
	    }
	    
	    for (Iterator kiteIterator = kiteMatches.iterator();
		 kiteIterator.hasNext();
		 ) {
		
		shape = ((Kite)kiteIterator.next()).getShape();
		output.drawShape(shape);
	    }
	    
	    if (exportColor) {
		output.setStrokeColor(dartsColor);
		output.setFillColor(dartsColor);
	    }
	    
	    for (Iterator dartIterator = dartMatches.iterator();
		 dartIterator.hasNext();
		 ) {
		
		shape = ((Dart)dartIterator.next()).getShape();
		output.drawShape(shape);
	    }
	    
	}
	else {
	    // don't fill shapes

	    if (exportColor) {
		output.setStrokeColor(kitesColor);
	    }
	    for (Iterator kiteIterator = kiteMatches.iterator();
		 kiteIterator.hasNext();
		 ) {
		
		shape = ((Kite)kiteIterator.next()).getShape();
		output.strokeShape(shape);
	    }
	    
	    if (exportColor) {
		output.setStrokeColor(dartsColor);
	    }

	    for (Iterator dartIterator = dartMatches.iterator();
		 dartIterator.hasNext();
		 ) {
		
		shape = ((Dart)dartIterator.next()).getShape();
		output.strokeShape(shape);
	    }
	    
	}
	if (drawIntersections) {
	
	    // draw points last, so they cover tiles
	    Iterator pointIterator = points.iterator();
	    
	    if (exportColor) {
		output.setStrokeColor(intersectionsColor);
	    }

	    while (pointIterator.hasNext()) {
		
		IntersectionPoint point = (IntersectionPoint)pointIterator.next();
		output.strokeShape(point.getShape());
	    }
	}
	
	return (System.currentTimeMillis() - time);

    }


    ///////////////////////////////
    // Getter and Setter Methods //
    ///////////////////////////////

    
    /**
     * Sets whether to draw forced bars or not
     *
     * @param newValue The new value to use
     */
    public void drawAxes(boolean newValue) {
	drawAxes = newValue;
    }

    /**
     * Sets whether to draw forced bars or not
     *
     * @param newValue The new value to use
     */
    public void drawForced(boolean newValue) {
	drawForced = newValue;
    }

    /**
     * Sets whether to draw unforced bars or not
     *
     * @param newValue The new value to use
     */
    public void drawUnforced(boolean newValue) {
	drawUnforced = newValue;
    }

    /**
     * Sets whether to draw intersections or not
     *
     * @param newValue The new value to use
     */
    public void drawIntersections(boolean newValue) {
	drawIntersections = newValue;
    }

    /**
     * Sets whether to draw Kite (thin rhomb) tiles or not
     *
     * @param newValue The new value to use
     */
    public void drawKites(boolean newValue) {
	drawKites = newValue;
    }

    /**
     * Sets whether to draw Dart (fat rhomb) tiles or not
     *
     * @param newValue The new value to use
     */
    public void drawDarts(boolean newValue) {
	drawDarts = newValue;
    }

    /**
     * Sets whether to fill tiles or not
     *
     * @param newValue The new value to use
     */
    public void fillTiles(boolean newValue) {
	fillTiles = newValue;
    }

    /**
     * Sets whether to export in color or not
     *
     * @param newValue The new value to use
     */
    public void exportColor(boolean newValue) {
	exportColor = newValue;
    }

    /**
     * Sets whether to mark the center or not
     *
     * @param newValue The new value to use
     */
    public void markCenter(boolean newValue) {
	markCenter = newValue;
    }

    /**
     * Sets whether to print tiles or rhombs
     *
     * @param newValue The new value to use
     */
    public void tilesNotRhombs(boolean newValue) {
	tilesNotRhombs = newValue;
    }

    /**
     * Gets whether to draw forced bars or not
     */
    public boolean drawAxes() {
	return drawAxes;
    }

    /**
     * Gets whether to draw forced bars or not
     */
    public boolean drawForced() {
	return drawForced;
    }

    /**
     * Gets whether to draw unforced bars or not
     */
    public boolean drawUnforced() {
	return drawUnforced;
    }

    /**
     * Gets whether to draw intersections or not
     */
    public boolean drawIntersections() {
	return drawIntersections;
    }

    /**
     * Gets whether to draw Kite (thin rhomb) tiles or not
     */
    public boolean drawKites() {
	return drawKites;
    }

    /**
     * Gets whether to draw Dart (fat rhomb) tiles or not
     */
    public boolean drawDarts() {
	return drawDarts;
    }

    /**
     * Gets whether to fill tiles or not
     */
    public boolean fillTiles() {
	return fillTiles;
    }

    /**
     * Gets whether to export in color or not
     */
    public boolean exportColor() {
	return exportColor;
    }

    /**
     * Gets whether to mark the center or not
     */
    public boolean markCenter() {
	return markCenter;
    }

    /**
     * Gets whether to print tiles or rhombs
     */
    public boolean tilesNotRhombs() {
	return tilesNotRhombs;
    }

    /**
     * Sets color for background
     *
     * @param newValue The new value to use
     */
    public void backgroundColor(Color newValue) {
	backgroundColor = newValue;
    }

    /**
     * Sets color for axes
     *
     * @param newValue The new value to use
     */
    public void axesColor(Color newValue) {
	axesColor = newValue;
    }

    /**
     * Sets color for forced bars
     *
     * @param newValue The new value to use
     */
    public void forcedColor(Color newValue) {
	forcedColor = newValue;
    }

    /**
     * Sets color for unforced bars
     *
     * @param newValue The new value to use
     */
    public void unforcedColor(Color newValue) {
	unforcedColor = newValue;
    }

    /**
     * Sets color for intersections
     *
     * @param newValue The new value to use
     */
    public void intersectionsColor(Color newValue) {
	intersectionsColor = newValue;
    }

    /**
     * Sets color for Kite (thin rhomb) tiles
     *
     * @param newValue The new value to use
     */
    public void kitesColor(Color newValue) {
	kitesColor = newValue;
    }

    /**
     * Sets color for Dart (fat rhomb) tiles
     *
     * @param newValue The new value to use
     */
    public void dartsColor(Color newValue) {
	dartsColor = newValue;
    }

    /**
     * Gets color for background
     */
    public Color backgroundColor() {
	return backgroundColor;
    }

    /**
     * Gets color for axes
     */
    public Color axesColor() {
	return	axesColor;
    }

    /**
     * Gets color for forced bars
     */
    public Color forcedColor() {
	return forcedColor;
    }

    /**
     * Gets color for unforced bars
     */
    public Color unforcedColor() {
	return unforcedColor;
    }

    /**
     * Gets color for intersections
     */
    public Color intersectionsColor() {
	return	intersectionsColor;
    }

    /**
     * Gets color for Kite (thin rhomb) tiles
     */
    public Color kitesColor() {
	return kitesColor;
    }

    /**
     * Gets color for Dart (fat rhomb) tiles
     */
    public Color dartsColor() {
	return dartsColor;
    }

    /**
     * Sets the filling transparency
     */
    public void setFillTransparency(double transparency) {

	fillTransparency = transparency;
	
	ALPHA_OVERLAY = 
	    AlphaComposite.getInstance(AlphaComposite.SRC_OVER, (float)transparency);
    }


    /////////////////////
    // Utility Functions
    /////////////////////


    /**
     * <p>
     * Determines if a distance is less than Epsilon (our margin of error)
     * </p>
     *
     * @param a The distance to compare
     *
     * @return boolean True, if the distance is smaller than the margin of error
     */
    public static boolean inRange(double a) {
	
	return inRange(0.0, a);

    }
    
    
    /**
     * <p>
     * Determines if two distances are within a certain tolerance (Epsilon)
     * of each other.
     * </p>
     *
     * @param a The first distance
     * @param b The second distance
     *
     * @return boolean True, if the two distances are within Epsilon of
     * each other.
     */
    public static boolean inRange(double a, double b) {

	return inRange(a, b, EPSILON);

    }
    
    
    /**
     * <p>
     * Determines if two distances are within the specified tolerance
     * of each other.
     * </p>
     *
     * @param a The first distance
     * @param b The second distance
     * @param epsilon The tolerance value that must be met.
     *
     * @return boolean True, if the two distances are within epsilon of
     * each other.
     */
    public static boolean inRange(double a, double b, double epsilon) {
	
	double difference = Math.abs(b - a);

	return (difference < epsilon);

    }
  
    
    /**
     * <p>
     * Returns sin(angle), where angle is in degrees.
     * </p>
     *
     * @param angle The angle, in degrees
     *
     * @return double sin(angle)
     */
    public static double sin(double angle)	{
	
	return Math.sin(Math.toRadians(angle));
    }
    

    /**
     * <p>
     * Returns cos(angle), where angle is in degrees.
     * </p>
     *
     * @param angle The angle, in degrees
     *
     * @return double cos(angle)
     */
    public static double cos(double angle)	{
	
	return Math.cos(Math.toRadians(angle));
    }
    

    /**
     * <p>
     * Returns tan(angle), where angle is in degrees.
     * </p>
     *
     * @param angle The angle, in degrees
     *
     * @return double tan(angle)
     */
    public static double tan(double angle)	{
	
	return Math.tan(Math.toRadians(angle));
    }
    

    /**
     * <p>
     * Returns the angle of elevation for any point, in degrees.
     * </p>
     *
     * @param x The x coordinate of the point
     * @param y The y coordinate of the point
     *
     * @return double The angle of elevation, as measured from the
     * x-axis, in degrees
     */
    public static double atan2(double x, double y)	{
	// Math.atan2 returns a degree in radians for a point, 
	// but only returns a value between -pi and pi.  Therefore,
	// we need to convert to degrees *and* make them positive

	// Math.atan2 expects args in reverse order
	return toDegrees(Math.atan2(y,x));

    }


    /**
     * <p>
     * Converts an angle from radians into <b>positive</b> degrees.
     * </p>
     *
     * @param a The angle to convert, in radians
     *
     * @return double The angle, where 0 <= angle < 360
     */
    public static double toDegrees(double a) {

	double temp = Math.toDegrees(a);

	if (temp < 0.0) {
	    temp += 360;
	}

	return temp;
    }


    /**
     * <p>
     * Forces an angle to be between 0 and 360 degrees.
     * </p>
     *
     * @param a The angle to normalize
     *
     * @return double The angle, where 0 <= angle < 360
     */
    public static double normalizeAngle(double a) {

	double temp = a;

	while (temp < 0) {
	    temp += 360;
	}

	while (temp >= 360) {
	    temp -= 360;
	}

	return temp;
    }
	

    /**
     * <p>
     * Sets the Penrose Tiling properties based on the properties object
     * provided.
     * </p>
     *
     * @param props The user-specified properties
     */
    public void loadProperties(Properties props) {

	// Debug
	try {
	    DEBUG = Integer.parseInt(props.getProperty("debug", ""));
	}
	catch (NumberFormatException nfe) {}


	// bounding box
	try {
	    double llx = Double.parseDouble(props.getProperty("llx", ""));
	    double lly = Double.parseDouble(props.getProperty("lly", ""));
	    double urx = Double.parseDouble(props.getProperty("urx", ""));
	    double ury = Double.parseDouble(props.getProperty("ury", ""));

	    double width = urx - llx;
	    double height = ury - lly;
	    double centerX = llx + (width / 2);
	    double centerY = lly + (height / 2);

	    setDimensions(width, height);
	    setOffset(centerX, centerY);

	}
	catch (NumberFormatException nfe) {
	    if (DEBUG > 0) {
		System.err.println("Invalid bounding box coordinates provided," +
				   " using defaults.");
	    }
	}

	// drawing variables
	if (props.getProperty("drawAxes", "true").equals("false")) {
	    drawAxes = false;
	}
	if (props.getProperty("drawForced", "true").equals("false")) {
	    drawForced = false;
	}
	if (props.getProperty("drawUnforced", "true").equals("false")) {
	    drawUnforced = false;
	}
	if (props.getProperty("drawIntersections", "true").equals("false")) {
	    drawIntersections = false;
	}
	if (props.getProperty("drawKites", "true").equals("false")) {
	    drawKites = false;
	}
	if (props.getProperty("drawDarts", "true").equals("false")) {
	    drawDarts = false;
	}
	if (props.getProperty("markCenter", "true").equals("false")) {
	    markCenter = false;
	}
	if (props.getProperty("fillTiles", "true").equals("false")) {
	    fillTiles = false;
	}
	if (props.getProperty("exportColor", "true").equals("false")) {
	    exportColor = false;
	}

	// Colors
	Color temp = null;
	
	temp = parseColor(props.getProperty("backgroundColor", ""));
	if (temp != null) {
	    backgroundColor = temp;
	}	
	temp = parseColor(props.getProperty("axesColor", ""));
	if (temp != null) {
	    axesColor = temp;
	}	
	temp = parseColor(props.getProperty("forcedColor", ""));
	if (temp != null) {
	    forcedColor = temp;
	}	
	temp = parseColor(props.getProperty("unforcedColor", ""));
	if (temp != null) {
	    unforcedColor = temp;
	}
	temp = parseColor(props.getProperty("intersectionsColor", ""));
	if (temp != null) {
	    intersectionsColor = temp;
	}
	temp = parseColor(props.getProperty("kitesColor", ""));
	if (temp != null) {
	    kitesColor = temp;
	}
	temp = parseColor(props.getProperty("dartsColor", ""));
	if (temp != null) {
	    dartsColor = temp;
	}
	
	// transparency
	try {
	    double alpha =
		Double.parseDouble(props.getProperty("fillTransparency", ""));
	    setFillTransparency(alpha);
	} catch (NumberFormatException nfe) {}
	
    }
	

    /**
     * <p>
     * Parses a string containing RGB color values (0-255) separated by
     * commas.  Returns null if it cannot parse the string.
     * </p>
     * 
     * @param rgbVals A string of comma-separated integers defining a
     * color in RGB color space.
     */
    public static Color parseColor(String rgbVals) {

	if (rgbVals == null) {
	    return null;
	}

	StringTokenizer tokenizer = new StringTokenizer(rgbVals, ",");
	
	int[] rgb = new int[3];

	for (int i = 0; i < 3; i++) {
	    try {
		rgb[i] = Integer.parseInt(tokenizer.nextToken());
	    }
	    catch (NumberFormatException nfe) {
		// couldn't parse the integer
		return null;
	    }
	    catch (java.util.NoSuchElementException nsee) {
		// not enough tokens to parse
		return null;
	    }
	}

	// return the new color from the parsed string
	return new Color(rgb[0], rgb[1], rgb[2]);

    }


    /**
     * <p>
     * Sets the Penrose Tiling properties based on the properties object
     * provided.
     * </p>
     *
     * @param props The user-specified properties
     */
    public void saveProperties(File file) throws IOException {

	Properties save = new Properties();

	save.setProperty("debug", ("" + DEBUG + ""));

	// transparency
	save.setProperty("fillTransparency", ("" + fillTransparency + ""));


	double[] bbox = getBoundingBox();

	save.setProperty("llx", ("" + bbox[0] + ""));
	save.setProperty("lly", ("" + bbox[1] + ""));
	save.setProperty("urx", ("" + bbox[2] + ""));
	save.setProperty("ury", ("" + bbox[3] + ""));

	save.setProperty("drawAxes", ("" + drawAxes + ""));
	save.setProperty("drawForced", ("" + drawForced + ""));
	save.setProperty("drawUnforced", ("" + drawUnforced + ""));
	save.setProperty("drawIntersections", ("" + drawIntersections + ""));
	save.setProperty("drawKites", ("" + drawKites + ""));
	save.setProperty("drawDarts", ("" + drawDarts + ""));
	save.setProperty("fillTiles", ("" + fillTiles + ""));
	save.setProperty("markCenter", ("" + markCenter + ""));
	save.setProperty("exportColor", ("" + exportColor + ""));

	// Colors
	save.setProperty("backgroundColor", printColor(backgroundColor));
	save.setProperty("axesColor", printColor(axesColor));
	save.setProperty("forcedColor", printColor(forcedColor));
	save.setProperty("unforcedColor", printColor(unforcedColor));
	save.setProperty("intersectionsColor", printColor(intersectionsColor));
	save.setProperty("kitesColor", printColor(kitesColor));
	save.setProperty("dartsColor", printColor(dartsColor));
	
	// write the properties out to the file
	FileOutputStream outStream = new FileOutputStream(file);
	save.store(outStream, "Penrose Tiling User Properties");

	
    }
	

    /**
     * <p>
     * Converts a color into a 4-integer comma delimited string, suitable
     * for reading in later.
     * </p>
     * 
     * @param color The color to convert to a string
     */
    protected static String printColor(Color color) {
	
	StringBuffer buffer = new StringBuffer(11);

	buffer.append(color.getRed());
	buffer.append(",");
	buffer.append(color.getGreen());
	buffer.append(",");
	buffer.append(color.getBlue());
	
	return buffer.toString();
    }


    /**
     * <p>
     * Runs the Penrose Tiling software from the command line, allowing
     * it to be scripted and run non-interactively.  Additionally, the
     * command line version is more memory efficient.
     * </p>
     *
     * @param args The command line arguments
     */
    public static void main(String[] args) {

	if (args.length > 2) {

	    File inFile = new File(args[0]);
	    File outFile = new File(args[1]);
	    
	    PenroseTiling pt = new PenroseTiling();
	    try {
		pt.loadConfiguration(inFile);
	    }
	    catch (IOException ioe) {
		System.err.println("Couldn't parse tiling configuration " +
				   ioe.getMessage());
		System.exit(1);
	    }

	    Properties userProperties;
	    
	    if (args.length > 3) {
		// get properties file, using system properties as default
		userProperties = new Properties(System.getProperties());
		
		try {
		    File propFile = new File(args[3]);
		    FileInputStream inStream = new FileInputStream(propFile);

		    userProperties.load(inStream);
		}
		catch (IOException ioe) {
		    System.err.println("Couldn't read properties file... skipping");
		    userProperties = System.getProperties();
		}
	    }
	    else {
		// just use the system properties
		userProperties = System.getProperties();
	    }
	    
	    pt.loadProperties(userProperties);

	    VectorOutput outputStream;

	    try {

		if (args[2].equalsIgnoreCase("pdf")) {
		    outputStream = pt.getPDFStream(outFile);
		}
		else {
		    outputStream = pt.getPostscriptStream(outFile);
		}
		
		pt.streamEmpire(outputStream);
		
		System.out.println("Stats:");
		
		int days, hours, minutes, seconds;
		
		days = (int)Math.floor(pt.time / 86400000);
		hours = (int)Math.floor((pt.time / 3600000) - (days * 24));
		minutes = (int)Math.floor((pt.time / 60000) - ((days * 24) + (hours * 60)));
		seconds = (int)Math.floor((pt.time / 1000) - ((days * 24) + (hours * 60) + (minutes *60)));

		System.out.println("\tTook " + days + " days, " +
				   hours + " hours, " +
				   minutes + " minutes, " +
				   seconds + " seconds.");

		System.out.println("\tFound " + pt.numTiles + " tiles");
		
	    }
	    catch (IOException ioe) {
		System.err.println("Couldn't Stream Penrose Empire: " +
				   ioe.getMessage());
	    }
	}
	else {
	    System.err.println("\nInsufficient number of arguments");
	    printUsage();
	}
    }
	

    /**
     * <p>
     * Prints the usage instructions for the command line interface.
     * </p>
     */
    public static void printUsage() {

	System.err.println("\n\nUsage:\njava PenroseTiling " +
			   "tiling-configuration output-file eps|pdf " +
			   "[properties-file]");
	
	System.err.println("\nSystem properties may be specified through " +
			   "the Java System Properties\ninterface, or by " +
			   "placing them in the properties file and " +
			   "specifying it on the\ncommand line.");

	System.err.println("\nproperties-file should contain valid " +
			   "system properties, which can be created\n" +
			   "and saved from within the Penrose Applet, or " +
			   "by editing a text file with the\nproperties " +
			   "in them.  Here is a list of user-definable " +
			   "properties:\n\n");

	System.err.println("debug                int: > 0 for debugging output");
	System.err.println("fillTransparency     double: 0 - 1");

	System.err.println("llx                  double: left edge of view");
	System.err.println("lly                  double: lower edge of view");
	System.err.println("urx                  double: right edge of view");
	System.err.println("ury                  double: upper edge of view");

	System.err.println("drawAxes             boolean: draw sequence axes");
	System.err.println("drawForced           boolean: draw forced bars");
	System.err.println("drawUnforced         boolean: draw unforced bars");
	System.err.println("drawIntersections    boolean: draw intersections");
	System.err.println("drawKites            boolean: draw Kite tiles");
	System.err.println("drawDarts            boolean: draw Dart tiles");
	System.err.println("fillTiles            boolean: fill tiles");
	System.err.println("markCenter           boolean: mark center of plane");
	System.err.println("exportColor          boolean: export/print in color");

	System.err.println("backgroundColor      color: background color");
	System.err.println("axesColor            color: axes color");
	System.err.println("forcedColor          color: forced bar color");
	System.err.println("unforcedColor        color: unforced bar color");
	System.err.println("intersectionsColor   color: intersection color");
	System.err.println("kitesColor           color: Kite tile color");
	System.err.println("dartsColor           color: Dart tile color");

	System.err.println("\n(All colors defined as 'rrr,ggg,bbb')\n\n");
	
	System.exit(1);
	
    }
	

}
