// $Header: /opt/cvs/java/net/logn/penrose/PointGraph.java,v 1.1 2001/02/05 20:23:10 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.penrose;

// Hashtable for primary points
import java.util.Hashtable;
import java.util.Enumeration;
// HashSet for secondary points
import java.util.HashSet;
import java.util.Iterator;


/**
 * <p>
 * A <b>PointGraph</b> is a structure for storing pairs of points.  It can
 * be visualized as a list of Points, each of which is associated with an
 * arbitrary number of points that are the correct distance away from it.
 * This structure prevents storing redundant points, and also ensures that
 * duplications do not occur.
 * </p>
 *
 * <p>
 * We say that we have a list of <b>primary</b> points, each of which has
 * at least one <b>secondary</b> point associated with it.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.1 $
 *
 * Last Modified $Date: 2001/02/05 20:23:10 $ by $Author: jhealy $
 */
public class PointGraph {

    /** Hash Table to store all the primary points */
    protected Hashtable table;


    /**
     * <p>
     * Constructs an empty PointGraph.
     * </p>
     */
    public PointGraph() {

	table = new Hashtable();

    }


    /**
     * <p>
     * Adds an association between the secondary and primary.
     * </p>
     *
     * @param primary The primary point to associate with
     * @param secondary The secondary point to associate
     */
    public void add(IntersectionPoint primary, IntersectionPoint secondary) {

	HashSet priList = (HashSet)table.get(primary);
	HashSet secList = (HashSet)table.get(secondary);

	// we cannot duplicate points, so we must make sure these points
	// have never been added

	if (secList == null) {
	    // just use the primary
	    if (priList == null) {
		priList = new HashSet();
		table.put(primary, priList);
	    }
	    priList.add(secondary);
	}
	else if (priList == null) {
	    // just use the secondary
	    secList.add(primary);
	}
	else {

	    // see if this point pairing already exists
	    if ( priList.contains(secondary) || secList.contains(primary) ) {
		// it does, so just return (no duplication)
		return;
	    }

	    // okay, we have both points as primary keys, but the pairing
	    // does not yet exist.  We'll have to decide which point should
	    // be the primary, and we do that by seeing which is "less than"
	    // the other.  The smaller point becomes the primary key.

	    if (secondary.compareTo(primary) < 0) {
		secList.add(primary);
	    }
	    else {
		priList.add(secondary);
	    }
	}

    }


    /**
     * <p>
     * Adds all the points from the given PointGraph to this PointGraph.
     * </p>
     *
     * @param graph The PointGraph to add to this one
     */
    public void addAll(PointGraph graph) {

	IntersectionPoint[] primaries = graph.getPrimaries();

	for (int i = 0; i < primaries.length; i++) {
	    
	    IntersectionPoint[] secondaries = graph.getSecondaries(primaries[i]);
	    
	    for (int j = 0; j < secondaries.length; j++) {
		
		add(primaries[i], secondaries[j]);
		
	    }
	    
	}

    }


    /**
     * <p>
     * Removes the primary point from the graph.
     * </p>
     *
     * @param primary The primary point to remove
     *
     * @return Collection The points that were associated with this primary
     */
    public IntersectionPoint removePrimary(IntersectionPoint primary) {

	return (IntersectionPoint)table.remove(primary);

    }


    /**
     * <p>
     * Removes the secondary point from the given primary association.
     * </p>
     *
     * @param primary The primary point to associate with
     * @param secondary The secondary point to associate
     *
     * @return IntersectionPoint The point that was removed
     */
    public IntersectionPoint removeSecondary(IntersectionPoint primary,
					     IntersectionPoint secondary) {

	HashSet secondaries = (HashSet)table.get(primary);

	boolean temp = secondaries.remove(secondary);
	
	// make sure we don't have empty associations
	if (secondaries.size() == 0) {
	    table.remove(primary);
	}

	if (temp) {
	    return secondary;
	}
	else {
	    return null;
	}

    }


    /**
     * <p>
     * Removes all the points in the given PointGraph.
     * </p>
     *
     * @param graph The PointGraph containing the points to be removed
     */
    public void removeAll(PointGraph graph) {

	IntersectionPoint[] primaries = graph.getPrimaries();

	for (int i = 0; i < primaries.length; i++) {

	    IntersectionPoint[] secondaries = graph.getSecondaries(primaries[i]);

	    for (int j = 0; j < secondaries.length; j++) {

		removeSecondary(primaries[i], secondaries[j]);
	    }

	}

    }


    /**
     * <p>
     * Returns an array containing the primary Intersection Points.
     * </p>
     *
     * @return IntersectionPoint[] Array of primary points
     */
    public IntersectionPoint[] getPrimaries() {

	IntersectionPoint[] points = new IntersectionPoint[table.size()];

	int index = 0;

	for (Enumeration keys = table.keys(); keys.hasMoreElements(); ) {

	    points[index] = (IntersectionPoint)keys.nextElement();
	    index++;

	}

	return points;

    }


    /**
     * <p>
     * Returns an array of all the secondary points for the given primary.
     * </p>
     *
     * @param primary The primary point to get secondary points from
     *
     * @return IntersectionPoint[] Array of secondary points for a primary
     */
    public IntersectionPoint[] getSecondaries(IntersectionPoint primary) {

	HashSet secondaries = (HashSet)table.get(primary);

	if (secondaries == null) {
	    return new IntersectionPoint[0];
	}

	IntersectionPoint[] points = new IntersectionPoint[secondaries.size()];
	int index = 0;

	for (Iterator values = secondaries.iterator();
	     values.hasNext();
	     ) {
	    
	    points[index] = (IntersectionPoint)values.next();
	    index++;

	}

	return points;
    }

}


