// $Header: /opt/cvs/java/net/logn/penrose/ZoomDialog.java,v 1.4 2001/02/26 04:47:23 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.penrose;

// Events (property changes are in beans)
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
// AWT Components
import java.awt.Frame;
// AWT Events
import java.awt.event.WindowEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
// Swing Components
import javax.swing.JOptionPane;
import javax.swing.JDialog;
import javax.swing.JTextField;



/**
 * <p>
 * <b>ZoomDialog</b> provides a modal dialog for the user to select the
 * zoom level to use when viewing a Penrose Tiling.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.4 $
 *
 * Last Modified $Date: 2001/02/26 04:47:23 $ by $Author: jhealy $
 */
public class ZoomDialog extends JDialog {

    // Text typed in by the user
    private String typedText = null;

    // Default value to use if user does not provide one
    private double defaultZoom;

    // Value that user has entered
    private double userZoom;

    // Dialog to present if the user provides invalid data
    private JOptionPane optionPane;


    /**
     * <p>
     * Returns the user's input, or the default if the user cancels
     * </p>
     *
     * @return double The zoom level that the user has entered
     */
    public double getValidatedZoom() {
        return userZoom;
    }


    /**
     * <p>
     * Constructs a ZoomDialog that will ask the user what zoom level
     * is desired and store the input value.
     * </p>
     *
     * @param defaultZoomLevel The zoom to use if the user does not enter one
     */
    public ZoomDialog(double defaultZoomLevel) {

	// Just use a bogus frame, since applets are not (exactly) frames
        super(new javax.swing.JFrame(), true);

	// Give this dialog a title bar
        setTitle("Set Zoom Level");

	// set the default zoom level
	defaultZoom = defaultZoomLevel;

	// Set the question to ask
        final String question = "What Zoom Factor would you like?";

	// Construct a text field with the default value already in it
        final JTextField textField = new JTextField(Double.toString(defaultZoom), 5);

	// Stick the visual components into an array
        Object[] visualArray = {question, textField};

	// Set the text of the two choice buttons
        final String zoomButton = "Set Zoom";
        final String cancelButton = "Cancel";

	// and put those buttons into an array
        Object[] buttonChoices = {zoomButton, cancelButton};

        optionPane = new JOptionPane(visualArray, 
                                    JOptionPane.QUESTION_MESSAGE,
                                    JOptionPane.YES_NO_OPTION,
                                    null,
                                    buttonChoices,
                                    buttonChoices[0]);

        setContentPane(optionPane);

	// close the window gracefully (cancel)
        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent we) {

		    // Instead of directly closing the window,
		    // we're going to change the JOptionPane's
		    // value property.

                    optionPane.setValue(new Integer(
						    JOptionPane.CLOSED_OPTION));
		}
	    });

	// when the user clicks, say what value is changing
        textField.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                optionPane.setValue(zoomButton);
            }
        });

        optionPane.addPropertyChangeListener(new PropertyChangeListener() {
		
		// Take action when the user clicks on the button
		public void propertyChange(PropertyChangeEvent e) {
		    String prop = e.getPropertyName();
		    
		    if (isVisible()  &&
			(e.getSource() == optionPane) &&
			(prop.equals(JOptionPane.VALUE_PROPERTY) ||
			 prop.equals(JOptionPane.INPUT_VALUE_PROPERTY))
			) {
			
			Object value = optionPane.getValue();
			
			if (value == JOptionPane.UNINITIALIZED_VALUE) {
			    //ignore reset
			    return;
			}
			
			// Reset the JOptionPane's value.
			// If you don't do this, then if the user
			// presses the same button next time, no
			// property change event will be fired.
			optionPane.setValue(
					    JOptionPane.UNINITIALIZED_VALUE);
			
			if (value.equals(zoomButton)) {
			    // If it was the zoom button that was clicked,
			    // get the text that the user typed in
                            typedText = textField.getText();

			    // try to parse it into a double
			    try {
				Double userValue = new Double(typedText);
				userZoom = userValue.doubleValue();
				// we got it, so now we can exit
				setVisible(false);
			    } catch (NumberFormatException nfe) {
                            // text was invalid
                            textField.selectAll();

			    // pop up an error message
                            JOptionPane.showMessageDialog(
                                            ZoomDialog.this,
                                            "Sorry, \"" + typedText + "\" "
                                            + "isn't a valid number.",
					    "Invalid Number",
                                            JOptionPane.ERROR_MESSAGE);
			    typedText = null;
			    }
                    } else { // user closed dialog or clicked cancel
			// set the zoom to the default and close
                        userZoom = defaultZoom;
                        typedText = null;
                        setVisible(false);
                    }
                }
            }
        });
    }
}






