// $Header: /opt/cvs/java/net/logn/util/vectoroutput/AbstractOutput.java,v 1.6 2001/02/15 14:18:23 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.util.vectoroutput;

// Hashtable to hold templates
import java.util.Hashtable;
// AWT Drawing classes
import java.awt.Shape;
import java.awt.Color;
import java.awt.Stroke;
// AffineTransforms for mapping
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.GeneralPath;
// File writing capabilities
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
// Truncating of numbers
import java.text.DecimalFormat;


/**
 * <p>
 * <b>AbstractOutput</b> defines some commonly used methods for dealing
 * with the saving of vector graphics.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.6 $
 *
 * Last Modified $Date: 2001/02/15 14:18:23 $ by $Author: jhealy $
 */
public abstract class AbstractOutput implements VectorOutput {

    /** OutputStream to write to */
    protected FileOutputStream outStream;

    /** Hashtable to hold the template shapes */
    protected Hashtable templates;

    /** Width of the page to output to (points or pixels) */
    protected double width;

    /** Height of the page to output to (points or pixels) */
    protected double height;

    /** Transformation to map drawing coordinates out to the output */
    protected AffineTransform transform;

    /** Bounding box: lower left x */
    protected double llx;

    /** Bounding box: lower left y */
    protected double lly;

    /** Bounding box: upper right x */
    protected double urx;

    /** Bounding box: upper right y */
    protected double ury;

    /** Stroke color */
    protected Color strokeColor = Color.black;

    /** Stroke transparency */
    protected double strokeTransparency = 1.0;

    /** Fill color */
    protected Color fillColor = Color.black;

    /** Fill transparency */
    protected double fillTransparency = 1.0;

    /** Initial Background Color */
    protected Color backgroundColor = null;

    /** Default formatter (5 significant digits) */
    public static final DecimalFormat truncator =
	new DecimalFormat("#####.########");

    /**
     * <p>
     * Constructor.  Creates a new empty Exporter object with the file 
     * specified. The bounding box defines the viewable portion of the
     * image.
     * </p>
     *
     * @param file The file to write out to
     * @param w The final width of the image
     * @param h The final height of the image
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     */
    protected AbstractOutput(File file, double w, double h,
			     double l, double b, double r, double t)
	throws IOException {

	this(file, null, w, h, l, b, r, t);
    }


    /**
     * <p>
     * Constructor.  Creates a new empty Exporter object with the file 
     * specified. The bounding box defines the viewable portion of the
     * image.
     * </p>
     *
     * @param file The file to write out to
     * @param bg The background color, or null for the default
     * @param w The final width of the image
     * @param h The final height of the image
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     */
    protected AbstractOutput(File file, Color bg, double w, double h,
			     double l, double b, double r, double t)
	throws IOException {

	outStream = new FileOutputStream(file);

	width = w;
	height = h;

	// get the transform for the drawing coordinates
	// fit to a single page, 612, 792
	transform = getMap(l, b, r, t, 0, 0, width, height);

	// Now map the user bounding box into the PDF space and store
	Point2D ill = new Point2D.Double(l, b);
	Point2D iur = new Point2D.Double(r, t);

	Point2D dll = transform.transform(ill, null);
	Point2D dur = transform.transform(iur, null);

	llx = dll.getX();
	lly = dll.getY();
	urx = dur.getX();
	ury = dur.getY();

	if (bg != null) {
	    backgroundColor = bg;
	}

	templates = new Hashtable();
    }


    /**
     * <p>
     * Prepares the file by writing any necessary header information.
     * </p>
     *
     */
    public abstract void prepare() throws IOException;


    /**
     * <p>
     * Finishes writing the file and closes the output file stream.
     * </p>
     *
     */
    public abstract void finish() throws IOException;


    /**
     * <p>
     * Sets the current stroking color.
     * </p>
     *
     * @param newColor The new stroking color
     */
    public abstract void setStrokeColor(Color newColor) throws IOException;


    /**
     * <p>
     * Sets the stroke width and type.
     * </p>
     *
     * @param stroke The new stroke
     */
    public abstract void setStroke(Stroke stroke) throws IOException;


    /**
     * <p>
     * Sets the stroke transparency (between 0 and 1)
     * </p>
     *
     * @param transparency The new transparency of the pen
     */
    public abstract void setStrokeTransparency(double transparency)
	throws IOException;


    /**
     * <p>
     * Sets the current filling color.
     * </p>
     *
     * @param newColor The new filling color
     */
    public abstract void setFillColor(Color newColor) throws IOException;


    /**
     * <p>
     * Sets the fill transparency (between 0 and 1)
     * </p>
     *
     * @param transparency The new transparency of the fill
     */
    public abstract void setFillTransparency(double transparency)
	throws IOException;


    /**
     * <p>
     * Strokes a shape to the output file.
     * </p>
     *
     * @param shape The shape to stroke
     */
    public abstract void strokeShape(Shape shape) throws IOException;


    /**
     * <p>
     * Strokes all the provided shapes to the output file.
     * </p>
     *
     * @param shapes The shapes to stroke
     */
    public void strokeShapes(Shape[] shapes) throws IOException {

	for (int i = 0; i < shapes.length; i++) {
	    strokeShape(shapes[i]);
	}

    }


    /**
     * <p>
     * Fills a shape to the output file.
     * </p>
     *
     * @param shape The shape to fill
     */
    public abstract void fillShape(Shape shape) throws IOException;


    /**
     * <p>
     * Fills all the provided shapes to the output file.
     * </p>
     *
     * @param shapes The shapes to fill
     */
    public void fillShapes(Shape[] shapes) throws IOException {

	for (int i = 0; i < shapes.length; i++) {
	    fillShape(shapes[i]);
	}

    }


    /**
     * <p>
     * Strokes and fills a shape to the output file.
     * </p>
     *
     * @param shape The shape to draw
     */
    public void drawShape(Shape shape) throws IOException {
	
	fillShape(shape);
	strokeShape(shape);
    }


    /**
     * <p>
     * Strokes and fills all the provided shapes to the output file.
     * </p>
     *
     * @param shapes The shapes to draw
     */
    public void drawShapes(Shape[] shapes) throws IOException {

	for (int i = 0; i < shapes.length; i++) {
	    drawShape(shapes[i]);
	}

    }


    /**
     * <p>
     * Creates a "template shape" that will be repeated often in the
     * image.  Subsequent calls can be made to the drawTemplate() and
     * fillTemplate() methods to fill these predefined shapes.
     * </p>
     *
     * @param name The name of the template
     * @param shape The shape to use for the template
     */
    public void createTemplate(String name, Shape shape) throws IOException {

	GeneralPath path = new GeneralPath(shape);

	templates.put(name, path);

    }


    /**
     * <p>
     * Gets the template shape with the given name, and applies the given
     * transform to it.
     * </p>
     *
     * @param name The name of the template to get
     * @param transform The transform to apply to the shape
     */
    protected Shape getTemplate(String name, AffineTransform transform) {

	GeneralPath path = (GeneralPath)templates.get(name);

	if (path != null) {
	    GeneralPath temp = (GeneralPath)path.clone();
	    temp.transform(transform);
	    return temp;
	}

	return null;

    }


    /**
     * <p>
     * Strokes the named template shape to the output file, using the transform
     * provided to place it in the correct position.
     * </p>
     *
     * @param name The name of the template to use
     * @param transform The transformation to apply to the template shape
     */
    public void strokeTemplate(String name, AffineTransform transform)
	throws IOException {

	Shape template = getTemplate(name, transform);

	if (template != null) {
	    drawShape(template);
	}

    }


    /**
     * <p>
     * Strokes the named template shape to the output file, using the
     * transforms provided to place them in the correct positions.
     * </p>
     *
     * @param name The name of the template to use
     * @param transforms The transformation to apply to the template shape
     */
    public void strokeTemplates(String name, AffineTransform[] transforms)
	throws IOException {

	for (int i = 0; i < transforms.length; i++) {
	    drawTemplate(name, transforms[i]);
	}
    }


    /**
     * <p>
     * Fills the named template shape to the output file, using the transform
     * provided to place it in the correct position.
     * </p>
     *
     * @param name The name of the template to use
     * @param transform The transformation to apply to the template shape
     */
    public void fillTemplate(String name, AffineTransform transform)
	throws IOException {

	Shape template = getTemplate(name, transform);
	
	if (template != null) {
	    fillShape(template);
	}

    }
    

    /**
     * <p>
     * Fills the named template shape to the output file, using the transforms
     * provided to place them in the correct positions.
     * </p>
     *
     * @param name The name of the template to use
     * @param transforms The transformation to apply to the template shape
     */
    public void fillTemplates(String name, AffineTransform[] transforms)
	throws IOException {

	for (int i = 0; i < transforms.length; i++) {
	    fillTemplate(name, transforms[i]);
	}
    }


    /**
     * <p>
     * Strokes and fills the named template shape to the output file, using
     * the transform provided to place it in the correct position.
     * </p>
     *
     * @param name The name of the template to use
     * @param transform The transformation to apply to the template shape
     */
    public void drawTemplate(String name, AffineTransform transform)
	throws IOException {

	Shape template = getTemplate(name, transform);

	if (template != null) {
	    drawShape(template);
	}

    }


    /**
     * <p>
     * Strokes and fills the named template shapes to the output file, 
     * using the transforms provided to place them in the correct positions.
     * </p>
     *
     * @param name The name of the template to use
     * @param transforms The transformation to apply to the template shape
     */
    public void drawTemplates(String name, AffineTransform[] transforms)
	throws IOException {

	for (int i = 0; i < transforms.length; i++) {
	    drawTemplate(name, transforms[i]);
	}
    }


    /**
     * <p>
     * Returns a transform that will scale and transform the current
     * drawing to the bounding size specified.  Proportions are maintained,
     * so the target region may not be completely filled.
     * </p>
     *
     * @param illx Input lower left x coordinate
     * @param illy Input lower left y coordinate
     * @param iurx Input upper right x coordinate
     * @param iury Input upper right y coordinate
     * @param dllx Device lower left x coordinate
     * @param dlly Device lower left y coordinate
     * @param durx Device upper right x coordinate
     * @param dury Device upper right y coordinate
     *
     * @return AffineTransform Transform that maps drawing cooridnates
     * to the output coordinates
     */
    public static AffineTransform getMap(double illx, double illy,
					 double iurx, double iury,
					 double dllx, double dlly,
					 double durx, double dury) {

	AffineTransform transform = new AffineTransform();

	double xScale = (durx - dllx) / (iurx - illx);
	double yScale = (dury - dlly) / (iury - illy);

	// this will hold the final scale param
	double scale;

	if (xScale < yScale) {
	    scale = xScale;
	}
	else {
	    scale = yScale;
	}

	transform.scale(scale, scale);

	transform.translate(-illx, -illy);

	return transform;
    }


    /**
     * <p>
     * Truncates a number so as not to exceed the precision of the output
     * device.
     * </p>
     *
     * @param number The number to truncate
     *
     * @return String The newly formatted number
     */
    public static String truncate(double number) {

	return truncator.format(number);

    }


}
