// $Header: /opt/cvs/java/net/logn/util/vectoroutput/PDF.java,v 1.6 2001/02/15 15:02:54 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.util.vectoroutput;

// AWT Drawing classes
import java.awt.Shape;
import java.awt.Color;
import java.awt.Stroke;
import java.awt.BasicStroke;
// AffineTransforms for mapping
import java.awt.geom.AffineTransform;
import java.awt.geom.PathIterator;
import java.awt.geom.Rectangle2D;
// File writing capabilities
import java.io.File;
import java.io.FileOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
// Compression classes to zip the output
import java.util.zip.Deflater;
import java.util.zip.DeflaterOutputStream;


/**
 * <p>
 * <b>PDF</b> writes vector graphics out to a PDF file.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.6 $
 *
 * Last Modified $Date: 2001/02/15 15:02:54 $ by $Author: jhealy $
 */
public class PDF extends AbstractOutput {


    /** Access to a printable version of the file stream */
    private DataOutputStream printer;

    /** Deflater object used to compress the PDF output */
    private Deflater compress;

    /** A compressed version of the file stream */
    private DeflaterOutputStream compressStream;

    /** Holds amount of bytes written out to the PDF stream (needed
	to construct PDF table of contents later) */
    private int[] bytesOut;

    /** PDF Title attribute */
    private String title;

    /** PDF Author attribute */
    private String author;

    /** PDF Subject attribute */
    private String subject;

    /** PDF Keywords attribute */
    private String keywords;

    /** PDF Creator attribute */
    private String creator;

    /** PDF Fill Command */
    private String F = "f";

    /** PDF Stroke Command */
    private String S = "S";

    /** PDF Stroke and Fill Command */
    private String FAS = "B";


    /**
     * <p>
     * Constructor.  Creates a new empty PDF exporter object with the file 
     * specified.  The image is scaled to fill a whole page.
     * </p>
     *
     * @param file The file to write out to
     * @param bg The background color, or null for the default
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     */
    public PDF(File file, Color bg, double l, double b, double r, double t)
	throws IOException {

	this(file, bg, l, b, r, t, null, null, null, null, null);
    }

	
    /**
     * <p>
     * Constructor.  Creates a new empty PDF exporter object with the file 
     * specified. The bounding box defines the viewable portion of the
     * image.
     * </p>
     *
     * @param file The file to write out to
     * @param bg The background color, or null for the default
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     * @param title The name to give to the file
     * @param author The author of this file
     * @param subject The subject of this file
     * @param keywords The keywords for this file
     * @param creator The creator application of this file
     */
    public PDF(File file, Color bg, double l, double b,
	       double r, double t, String title, String author,
	       String subject, String keywords, String creator)
	throws IOException {

	// a full letter page in PDF is 612 x 792 points
	this(file, bg, 612.0, 792.0, l, b, r, t,
	     title, author, subject, keywords, creator);

    }


    /**
     * <p>
     * Constructor.  Creates a new empty PDF exporter object with the file 
     * specified.  The image is scaled to the point sizes specified.
     * </p>
     *
     * @param file The file to write out to
     * @param bg The background color, or null for the default
     * @param w The width of the final image
     * @param h The height of the final image
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     */
    public PDF(File file, Color bg, double w, double h, 
	       double l, double b, double r, double t) throws IOException {
	

	this(file, bg, w, h, l, b, r, t, null, null, null, null, null);

    }


    /**
     * <p>
     * Constructor.  Creates a new empty PDF exporter object with the file 
     * specified. The bounding box defines the viewable portion of the
     * image.
     * </p>
     *
     * @param file The file to write out to
     * @param bg The background color, or null for the default
     * @param w The width of the final image
     * @param h The height of the final image
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     * @param title The name to give to the file
     * @param author The author of this file
     * @param subject The subject of this file
     * @param keywords The keywords for this file
     * @param creator The creator application of this file
     */
    public PDF(File file, Color bg, double w, double h, double l, double b,
	       double r, double t, String title, String author,
	       String subject, String keywords, String creator)
	throws IOException {

	super(file, bg, w, h, l, b, r, t);

	this.title = title;
	this.author = author;
	this.subject = subject;
	this.keywords = keywords;
	this.creator = creator;

	// Initialize the table of contents byte array.
	bytesOut = new int[8];
	bytesOut[0] = 0;

	// Initialize the compression utilities
	compress = new Deflater(Deflater.BEST_COMPRESSION);

    }


    /**
     * <p>
     * Prepares the file by writing any necessary header information.
     * </p>
     *
     */
    public void prepare() throws IOException {

	// construct the printable objects
	printer = new DataOutputStream(outStream);

	// print PDF Header
	printer.writeBytes("%PDF-1.3\n");

	// Put 4 non-ascii chars here to mark the file as binary
	// (recommended by PDF spec) I use the four chars 254
	printer.writeBytes("%");
	printer.write(0xde);
	printer.write(0xad);
	printer.write(0xbe);
	printer.write(0xef);
	printer.writeBytes("\n");

	bytesOut[0] = printer.size();

	// Catalog
	printer.writeBytes("1 0 obj\n");
	printer.writeBytes("  <<\n");
	printer.writeBytes("     /Type /Catalog\n");
	printer.writeBytes("     /Outlines 2 0 R\n");
	printer.writeBytes("     /Pages 3 0 R\n");
	printer.writeBytes("  >>\n");
	printer.writeBytes("endobj\n\n");

	bytesOut[1] = printer.size();

	// Outline
	printer.writeBytes("2 0 obj\n");
	printer.writeBytes("  <<\n");
	printer.writeBytes("     /Type /Outlines\n");
	printer.writeBytes("     /Count 0\n");
	printer.writeBytes("  >>\n");
	printer.writeBytes("endobj\n\n");

	bytesOut[2] = printer.size();

	// Pages
	printer.writeBytes("3 0 obj\n");
	printer.writeBytes("  <<\n");
	printer.writeBytes("     /Type /Pages\n");
	printer.writeBytes("     /Kids [4 0 R]\n");
	printer.writeBytes("     /Count 1\n");
	printer.writeBytes("  >>\n");
	printer.writeBytes("endobj\n\n");

	bytesOut[3] = printer.size();

	// Page
	printer.writeBytes("4 0 obj\n");
	printer.writeBytes("  <<\n");
	printer.writeBytes("     /Type /Page\n");
	printer.writeBytes("     /Parent 3 0 R\n");
	printer.writeBytes("     /MediaBox [" +
			(int)Math.floor(llx) + " " +
			(int)Math.floor(lly) + " " +
			(int)Math.ceil(urx) + " " +
			(int)Math.ceil(ury) + "]\n");
	printer.writeBytes("     /Contents 5 0 R\n");
	printer.writeBytes("     /Resources << /ProcSet [/PDF] >>\n");
	printer.writeBytes("  >>\n");
	printer.writeBytes("endobj\n\n");

	bytesOut[4] = printer.size();

	// Content Stream
	printer.writeBytes("5 0 obj\n");
	printer.writeBytes("  << /Length 6 0 R /Filter /FlateDecode >>\n");
	printer.writeBytes("stream\n");

	// remeber how many bytes we've written
	bytesOut[5] = printer.size();

	// begin actual graphics code

	// we compress the graphics, so crank up the compressed stream
	printer.flush();
	compressStream = new DeflaterOutputStream(outStream, compress);
	
	printer = new DataOutputStream(compressStream);

	// Graphics Initialization

	// save the current graphics state
	printer.writeBytes("q\n");

	if (backgroundColor != null) {
	    // set the background color
	    setFillColor(backgroundColor);
	    // draw a rectangle that fills the whole screen
	    fillShape(new Rectangle2D.Double(0, 0, width, height));
	}
	else {
	    backgroundColor = Color.white;
	}

    }


    /**
     * <p>
     * Finishes writing the file and closes the output file stream.
     * </p>
     *
     */
    public void finish() throws IOException {

	// restore the graphics state
	printer.writeBytes("Q\n");

	compressStream.finish();

	// get the new offset value to use after the graphics stream
	int offset = compress.getTotalOut() + bytesOut[5];

	// switch back to the uncompressed stream
	printer = new DataOutputStream(outStream);

	printer.writeBytes("endstream\n");
	printer.writeBytes("endobj\n\n");

	bytesOut[5] = offset + printer.size();

	// Size of graphics object
	printer.writeBytes("6 0 obj\n");
	printer.writeBytes(compress.getTotalOut() + "\n");
	printer.writeBytes("endobj\n\n");

	bytesOut[6] = offset + printer.size();

	// Doc metadata
	printer.writeBytes("7 0 obj\n");
	printer.writeBytes("  <<\n");
	if (title != null) {
	    printer.writeBytes("     /Title (" + title + ")\n");
	}
	if (author != null) {
	    printer.writeBytes("     /Author (" + author + ")\n");
	}
	if (subject != null) {
	    printer.writeBytes("     /Subject (" + subject + ")\n");
	}
	if (keywords != null) {
	    printer.writeBytes("     /Keywords (" + keywords + ")\n");
	}
	if (creator != null) {
	    printer.writeBytes("     /Creator (" + creator + ")\n");
	}
	printer.writeBytes("     /Producer (LogN Java Shape to PDF Converter)\n");

	// get the current date and time
	java.text.SimpleDateFormat formatter =
	    new java.text.SimpleDateFormat ("yyyyMMddHHmmss");
	String dateString = formatter.format(new java.util.Date());
	
	printer.writeBytes("     /CreationDate (D:" + dateString + ")\n");

	printer.writeBytes("  >>\n");
	printer.writeBytes("endobj\n\n");


	bytesOut[7] = offset + printer.size();
	
	// Construct the table of contents

	printer.writeBytes("xref\n");
	printer.writeBytes("0 " + bytesOut.length + "\n");
	printer.writeBytes("0000000000 65535 f \n");

	for (int i = 0; i < (bytesOut.length - 1); i++) {
	    StringBuffer digits = new StringBuffer();
	    digits.append(bytesOut[i]);

	    for (int j = 0; j < (10 - digits.length()); j++) {
		printer.writeBytes("0");
	    }

	    printer.writeBytes(digits.toString() + " 00000 n \n");
	}

	// Trailer info
	printer.writeBytes("trailer\n");
	printer.writeBytes("  <<\n");
	printer.writeBytes("    /Size " + bytesOut.length + "\n");
	printer.writeBytes("    /Root 1 0 R\n");
	printer.writeBytes("    /Info 7 0 R\n");
	printer.writeBytes("  >>\n");

	// Static reference section
	printer.writeBytes("startxref\n");
	printer.writeBytes(bytesOut[(bytesOut.length - 1)] + "\n");

	printer.writeBytes("%%EOF\n");

    }


    /**
     * <p>
     * Sets a color.
     * </p>
     *
     * @param color The color to set
     * @param alpha The transparency to apply to the color
     */
    protected void setColor(Color color, double alpha) throws IOException {

	// postscript doesn't really "do" transparency, so we'll fake it
	// by mixing the paint color with the current background color.
	// Note that this does not make layered objects transparent; it
	// only makes objects transparent relative to the current background.

	float[] fg = color.getRGBColorComponents(null);
	float[] bg = backgroundColor.getRGBColorComponents(null);
	double[] trans = new double[3];
	
	for (int i = 0; i < trans.length; i++) {
	    trans[i] = (alpha * fg[i]) + ((1 - alpha) * bg[i]);
	}

	printer.writeBytes(trans[0] + " " + trans[1] + " " +
			trans[2] + " ");

    }


    /**
     * <p>
     * Sets the current stroking color.
     * </p>
     *
     * @param newColor The new stroking color
     */
    public void setStrokeColor(Color newColor) throws IOException {

	strokeColor = newColor;

	setColor(strokeColor, strokeTransparency);

	printer.writeBytes("RG\n");

    }


    /**
     * <p>
     * Sets the current filling color.
     * </p>
     *
     * @param newColor The new filling color
     */
    public void setFillColor(Color newColor) throws IOException {

	fillColor = newColor;

	setColor(fillColor, fillTransparency);

	printer.writeBytes("rg\n");

    }


    /**
     * <p>
     * Sets the pen stroke width and type.
     * </p>
     *
     * @param stroke The new stroke
     */
    public void setStroke(Stroke stroke) throws IOException {

	BasicStroke s = (BasicStroke)stroke;

	// 0 == hairline
	printer.writeBytes(s.getLineWidth() + " w\n");

	// get the cap type
	int cap = s.getEndCap();
	int myCap = 0;

	switch (cap) {
            
	case BasicStroke.CAP_BUTT:
	    myCap = 0;
	    break;
		
	case BasicStroke.CAP_ROUND:
	    myCap = 1;
	    break;
		
	case BasicStroke.CAP_SQUARE:
	    myCap = 2;
	    break;
	}
	printer.writeBytes(myCap + " J\n");

	// get the line join
	int line = s.getLineJoin();
	int myLine = 0;

	switch (line) {
            
	case BasicStroke.JOIN_MITER:
	    myLine = 0;
	    break;
	
	case BasicStroke.JOIN_ROUND:
	    myLine = 1;
	    break;
		
	case BasicStroke.JOIN_BEVEL:
	    myLine = 2;
	    break;
	}
	printer.writeBytes(myLine + " j\n");

	// get miter limit
	printer.writeBytes(s.getMiterLimit() + " M\n");

	// get the dash
	float[] dash = s.getDashArray();

	if (dash != null) {
	    printer.writeBytes("[");
	    for (int i = 0; i < dash.length; i++) {
		printer.writeBytes(dash[i] + " ");
	    }
	    printer.writeBytes("] " + s.getDashPhase() + " d\n");
	}
	else {
	    printer.writeBytes("[] 0 d\n");
	}
    }


    /**
     * <p>
     * Sets the stroke transparency (between 0 and 1).
     * </p>
     *
     * @param transparency The new transparency of the pen
     */
    public void setStrokeTransparency(double transparency) throws IOException {

	strokeTransparency = transparency;
	setStrokeColor(strokeColor);
    }


    /**
     * <p>
     * Sets the fill transparency (between 0 and 1).
     * </p>
     *
     * @param transparency The new transparency of the pen
     */
    public void setFillTransparency(double transparency) throws IOException {

	fillTransparency = transparency;
	setFillColor(fillColor);
    }


    /**
     * <p>
     * Draws a shape to the output file.
     * </p>
     *
     * @param shape The shape to draw
     */
    protected void outputPath(Shape shape) throws IOException {

	// get a path iterator over the shape
	PathIterator pi = shape.getPathIterator(transform);

	// Get the winding rule, and set the fill type as needed
	int winding = pi.getWindingRule();

	if (winding == PathIterator.WIND_EVEN_ODD) {
	    F = "f*";
	    FAS = "B*";
	}
	else {
	    // assume nonzero winding rule
	    F = "f";
	    FAS = "B";
	}

	// now iterate over every path in the shape
	while (!pi.isDone()) {
	    
	    // construct an array to hold the coordinates of this segment
	    double[] points = new double[6];
	    // integer to hold the segment type
	    int segType;

	    segType = pi.currentSegment(points);

	    // switch on the segment type
	    switch (segType) {
		
	    case PathIterator.SEG_MOVETO:
		printer.writeBytes(truncate(points[0]) + " " +
				   truncate(points[1]) + " m\n");
		break;
		
	    case PathIterator.SEG_LINETO:
		printer.writeBytes(truncate(points[0]) + " " +
				   truncate(points[1]) + " l\n");
		break;
		
	    case PathIterator.SEG_CUBICTO:
		printer.writeBytes(truncate(points[0]) + " " +
				   truncate(points[1]) + " " +
				   truncate(points[2]) + " " +
				   truncate(points[3]) + " " +
				   truncate(points[4]) + " " +
				   truncate(points[5]) + " c\n");
		break;
		
	    case PathIterator.SEG_QUADTO:
		System.err.println("Quadric Curve " +
				   points[0] + " " + points[1] + " " +
				   points[2] + " " + points[3] +
				   " NOT IMPLEMENTED");
		break;
		
	    case PathIterator.SEG_CLOSE:
		printer.writeBytes("h\n");
		break;
		
	    }

	    pi.next();

	}

    }


    /**
     * <p>
     * Strokes a shape to the output file.
     * </p>
     *
     * @param shape The shape to stroke
     */
    public void strokeShape(Shape shape) throws IOException {

	outputPath(shape);
	printer.writeBytes(S + "\n");

    }


    /**
     * <p>
     * Fills a shape to the output file.
     * </p>
     *
     * @param shape The shape to fill
     */
    public void fillShape(Shape shape) throws IOException {
	
	outputPath(shape);
	printer.writeBytes(F + "\n");

    }


    /**
     * <p>
     * Strokes and fills a shape to the output file.
     * </p>
     *
     * @param shape The shape to draw
     */
    public void drawShape(Shape shape) throws IOException {

	outputPath(shape);
	printer.writeBytes(FAS + "\n");

    }


    public static void main(String[] args) {

	File test = new File(args[0]);

	try {

	    PDF pdf = new PDF(test, null, 612.0, 792.0, 100.0, 100.0, 400.0, 400.0,
			      "Test of PDF code",
			      "Jason Healy",
			      "This is a test",
			      "test pdf",
			      "The main() method of the PDF code");
	    
	    pdf.prepare();

	    pdf.setStroke(new BasicStroke());

	    pdf.setStrokeColor(Color.black);
	    pdf.strokeShape(new java.awt.geom.Rectangle2D.Double(110,110,280,280));

	    pdf.setFillColor(Color.blue);
	    pdf.fillShape(new java.awt.geom.Rectangle2D.Double(150,150,100,200));

	    pdf.drawShape(new java.awt.geom.Ellipse2D.Double(120,120,200,100));

	    pdf.finish();
	}
	catch (IOException ioe) {
	    System.err.println("Caught IO Exception" + ioe.getMessage());
	    ioe.printStackTrace();
	}
    }

}

