// $Header: /opt/cvs/java/net/logn/util/vectoroutput/PNG.java,v 1.5 2001/02/19 17:01:56 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.util.vectoroutput;

// AWT Drawing classes
import java.awt.Shape;
import java.awt.Color;
import java.awt.image.BufferedImage;
import java.awt.Graphics2D;
// AffineTransforms for mapping
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
// File writing capabilities
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
// Imaging API
import javax.media.jai.JAI;
//import javax.media.jai.RenderedOp;
import com.sun.media.jai.codec.PNGEncodeParam;
// Rendering options
import java.awt.Stroke;
import java.awt.BasicStroke;
import java.awt.AlphaComposite;
import java.awt.RenderingHints;


/**
 * <p>
 * <b>PNG</b> writes vector graphics out to a PNG file.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.5 $
 *
 * Last Modified $Date: 2001/02/19 17:01:56 $ by $Author: jhealy $
 */
public class PNG extends AbstractOutput {

    /** Image to render shapes to */
    private BufferedImage renderedImage;

    /** Graphics derrived from the image */
    private Graphics2D graphics;


    /**
     * <p>
     * Constructor.  Creates a new empty PNG object with the file specified.
     * The bounding box defines the viewable portion of the image; it will
     * be scaled to fit within the width and height parameters specified.
     * </p>
     *
     * @param file The file to write out to
     * @param bg The background color, or null for the default
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     */
    public PNG(File file, Color bg, double l, double b, double r, double t)
	throws IOException {
	
	super(file, bg, (r - l), (t - b), l, b, r, t);

    }


    /**
     * <p>
     * Constructor.  Creates a new empty PNG object with the file specified.
     * The bounding box defines the viewable portion of the image; it will
     * be scaled to fit within the width and height parameters specified.
     * </p>
     *
     * @param file The file to write out to
     * @param bg The background color, or null for the default
     * @param w The width of the image
     * @param h The height of the image
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     */
    public PNG(File file, Color bg, double w, double h, double l, double t,
	       double r, double b) throws IOException {

	super(file, bg, w, h, l, b, r, t);

    }


    /**
     * <p>
     * Prepares the file by writing any necessary header information.
     * </p>
     *
     */
    public void prepare() throws IOException {

	renderedImage = new BufferedImage((int)Math.ceil(width),
					  (int)Math.ceil(height),
					  BufferedImage.TYPE_INT_RGB);

	graphics = renderedImage.createGraphics();

	// set the background color, if it has been specified
	if (backgroundColor != null) {
	    setFillColor(backgroundColor);
	}
	else {
	    backgroundColor = Color.white;
	}

	// draw a rectangle that fills the whole screen, clearing it
	// to the background color.
	graphics.setBackground(backgroundColor);
	graphics.fill(new Rectangle2D.Double(0, 0,
					     Math.ceil(width),
					     Math.ceil(height)));


	graphics.setColor(strokeColor);

	// now, apply the transform so that the image is centered inside the
	// PNG.  Because Java uses an "upside-down" notion of graphics (-y is
	// "up" and +y is "down"), we have to munge the matrix a little:

	transform.translate((-2 *
			     (transform.getTranslateX() / transform.getScaleX())
			     ), 0.0);

	// Now we apply that transform to the graphics object, and we're done
	graphics.setTransform(transform);

	RenderingHints rHints = new RenderingHints(RenderingHints.KEY_ANTIALIASING,
						   RenderingHints.VALUE_ANTIALIAS_ON);
	
	rHints.put(RenderingHints.KEY_INTERPOLATION,
		   RenderingHints.VALUE_INTERPOLATION_BICUBIC);

	rHints.put(RenderingHints.KEY_RENDERING,
		   RenderingHints.VALUE_RENDER_QUALITY);
	
	rHints.put(RenderingHints.KEY_COLOR_RENDERING,
		   RenderingHints.VALUE_COLOR_RENDER_QUALITY);

	graphics.setRenderingHints(rHints);

    }


    /**
     * <p>
     * Finishes writing the file and closes the output file stream.
     * </p>
     *
     */
    public void finish() throws IOException {

	JAI.create("encode", renderedImage, outStream, "PNG",
		   (PNGEncodeParam.getDefaultEncodeParam(renderedImage)));

    }


    /**
     * <p>
     * Sets the current stroking color.
     * </p>
     *
     * @param newColor The new stroking color
     */
    public void setStrokeColor(Color newColor) {

	strokeColor = newColor;

    }


    /**
     * <p>
     * Sets the current filling color.
     * </p>
     *
     * @param newColor The new filling color
     */
    public void setFillColor(Color newColor) {
	
	fillColor = newColor;

    }


    /**
     * <p>
     * Sets the pen stroke width and type.
     * </p>
     *
     * @param stroke The new stroke
     */
    public void setStroke(Stroke stroke) {

	graphics.setStroke(stroke);
    }


    /**
     * <p>
     * Sets the stroke transparency (between 0 and 1).
     * </p>
     *
     * @param transparency The new transparency of the pen
     */
    public void setStrokeTransparency(double transparency) {

	strokeTransparency = transparency;
    }


    /**
     * <p>
     * Sets the fill transparency (between 0 and 1).
     * </p>
     *
     * @param transparency The new transparency of the pen
     */
    public void setFillTransparency(double transparency) {

	fillTransparency = transparency;
    }


    /**
     * <p>
     * Strokes a shape to the output file.
     * </p>
     *
     * @param shape The shape to stroke
     */
    public void strokeShape(Shape shape) throws IOException {

	graphics.setColor(strokeColor);

	AlphaComposite alpha =
	    AlphaComposite.getInstance(AlphaComposite.SRC_OVER,
				       (float)strokeTransparency);

	graphics.setComposite(alpha);
	
	graphics.draw(shape);

    }


    /**
     * <p>
     * Fills a shape to the output file.
     * </p>
     *
     * @param shape The shape to fill
     */
    public void fillShape(Shape shape) throws IOException {
	
	graphics.setColor(fillColor);

	AlphaComposite alpha =
	    AlphaComposite.getInstance(AlphaComposite.SRC_OVER,
				       (float)fillTransparency);

	graphics.setComposite(alpha);
	
	graphics.fill(shape);

    }


    public static void main(String[] args) {

	File test = new File(args[0]);

	try {
	    PNG png = new PNG(test, null, 10, 10, 100, 450);
	    
	    png.prepare();

	    png.setFillColor(Color.red);
	    png.fillShape(new java.awt.geom.Rectangle2D.Double(15,15,20,420));

	    png.setStrokeColor(Color.blue);
	    png.strokeShape(new java.awt.geom.Ellipse2D.Double(20,20,10,410));

	    png.drawShape(new java.awt.geom.Ellipse2D.Double(40,40,40,200));

	    png.finish();
	}
	catch (IOException ioe) {
	    System.err.println("Caught IO Exception" + ioe.getMessage());
	    ioe.printStackTrace();
	}
    }

}
