// $Header: /opt/cvs/java/net/logn/util/vectoroutput/Postscript.java,v 1.11 2001/02/14 22:55:26 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.util.vectoroutput;

// AWT Drawing classes
import java.awt.Shape;
import java.awt.Color;
import java.awt.Stroke;
import java.awt.BasicStroke;
// AffineTransforms for mapping
import java.awt.geom.AffineTransform;
import java.awt.geom.PathIterator;
import java.awt.geom.Rectangle2D;
// File writing capabilities
import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.io.IOException;


/**
 * <p>
 * <b>Postscript</b> writes vector graphics out to an Encapsulated
 * Postscript file.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.11 $
 *
 * Last Modified $Date: 2001/02/14 22:55:26 $ by $Author: jhealy $
 */
public class Postscript extends AbstractOutput {


    /** Access to a printable version of the file stream */
    private PrintStream printer;

    /** EPS Title comment value */
    private String title;

    /** EPS Author comment value */
    private String author;

    /** EPS Subject comment value */
    private String subject;

    /** EPS Keywords comment value */
    private String keywords;

    /** EPS Creator comment value */
    private String creator;

    /** Postscript Fill Command */
    private String F = "fill";

    /** Postscript Stroke Command */
    private String S = "stroke";

    /** Internal index for numbering user functions */
    private int functionIndex = 1;

    /** The current paint color (used to remove duplicate color changes) */
    private Color currentColor = null;

    /** The current transparency (used to remove duplicate alpha changes) */
    private double currentTransparency = -1.0;


    /**
     * <p>
     * Constructor.  Creates a new empty EPS exporter object with the file 
     * specified.  The image is scaled to fill a whole page.
     * </p>
     *
     * @param file The file to write out to
     * @param bg The background color, or null for the default
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     */
    public Postscript(File file, Color bg, double l, double b, double r, double t)
	throws IOException {

	this(file, bg, l, b, r, t, null, null, null, null, null);
    }

	
    /**
     * <p>
     * Constructor.  Creates a new empty EPS exporter object with the file 
     * specified. The bounding box defines the viewable portion of the
     * image.
     * </p>
     *
     * @param file The file to write out to
     * @param bg The background color, or null for the default
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     * @param title The name to give to the file
     * @param author The author of this file
     * @param subject The subject of this file
     * @param keywords The keywords for this file
     * @param creator The creator application of this file
     */
    public Postscript(File file, Color bg, double l, double b,
		      double r, double t, String title, String author,
		      String subject, String keywords, String creator)
	throws IOException {

	// a full letter page is 612 x 792 points
	this(file, bg, 612.0, 792.0, l, b, r, t,
	     title, author, subject, keywords, creator);

    }


    /**
     * <p>
     * Constructor.  Creates a new empty EPS exporter object with the file 
     * specified.  The image is scaled to the point sizes specified.
     * </p>
     *
     * @param file The file to write out to
     * @param bg The background color, or null for the default
     * @param w The width of the final image
     * @param h The height of the final image
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     */
    public Postscript(File file, Color bg, double w, double h, 
	       double l, double b, double r, double t) throws IOException {
	

	this(file, bg, w, h, l, b, r, t, null, null, null, null, null);

    }


    /**
     * <p>
     * Constructor.  Creates a new empty EPS exporter object with the file 
     * specified. The bounding box defines the viewable portion of the
     * image.
     * </p>
     *
     * @param file The file to write out to
     * @param bg The background color, or null for the default
     * @param w The width of the final image
     * @param h The height of the final image
     * @param l The x coordinate of the left side of the bounding box
     * @param b The y coordinate of the bottom side of the bounding box
     * @param r The x coordinate of the right side of the bounding box
     * @param t The y coordinate of the top side of the bounding box
     * @param title The name to give to the file
     * @param author The author of this file
     * @param subject The subject of this file
     * @param keywords The keywords for this file
     * @param creator The creator application of this file
     */
    public Postscript(File file, Color bg, double w, double h, double l, double b,
	       double r, double t, String title, String author,
	       String subject, String keywords, String creator)
	throws IOException {

	super(file, bg, w, h, l, b, r, t);

	this.title = title;
	this.author = author;
	this.subject = subject;
	this.keywords = keywords;
	this.creator = creator;

    }


    /**
     * <p>
     * Prepares the file by writing any necessary header information.
     * </p>
     *
     */
    public void prepare() throws IOException {

	printer = new PrintStream(outStream);

	// print EPS Header
	printer.println("%!PS-Adobe-3.0 EPSF-3.0");
	if (title != null) {
	    printer.println("%%Title (" + title + ")");
	}
	if (author != null) {
	    printer.println("%%Author (" + author + ")");
	}
	if (subject != null) {
	    printer.println("%%Subject (" + subject + ")");
	}
	if (keywords != null) {
	    printer.println("%%Keywords (" + keywords + ")");
	}
	if (creator != null) {
	    printer.println("%%Creator (" + creator + ")");
	}
	printer.println("%%Producer (LogN Java Shape to PDF Converter)");

	// get the current date and time
	java.text.SimpleDateFormat formatter =
	    new java.text.SimpleDateFormat ("yyyyMMddHHmmss");
	String dateString = formatter.format(new java.util.Date());
	
	printer.println("%%CreationDate (D:" + dateString + ")");
	
	printer.println("%%DocumentData: Clean7Bit");
	printer.println("%%Origin: 0 0");
	printer.println("%%BoundingBox: " +
			(int)Math.floor(llx) + " " +
			(int)Math.floor(lly) + " " +
			(int)Math.ceil(urx) + " " +
			(int)Math.ceil(ury));
	printer.println("%%HiresBoundingBox: " + llx + " " + lly +
			" " + urx + " " + ury);
	printer.println("%%ExactBoundingBox: " + llx + " " + lly +
			" " + urx + " " + ury);
	printer.println("%%LanguageLevel: 2");
	printer.println("%%Pages: 1");
	printer.println("%%Page: 1");
	printer.println("%%EndComments\n");

	// save the current graphics state
	printer.println("\ngsave");

	// clip to the provided graphics
	printer.println("newpath");
	printer.println(llx + " " + lly + " moveto");
	printer.println(urx + " " + lly + " lineto");
	printer.println(urx + " " + ury + " lineto");
	printer.println(llx + " " + ury + " lineto");
	printer.println("closepath");
	printer.println("clip");

	// set the background color
	if (backgroundColor != null) {
	    // set the background color
	    setFillColor(backgroundColor);
	    // draw a rectangle that fills the whole screen
	    fillShape(new Rectangle2D.Double(0, 0, width, height));
	}
	else {
	    backgroundColor = Color.white;
	}


    }


    /**
     * <p>
     * Finishes writing the file and closes the output file stream.
     * </p>
     *
     */
    public void finish() throws IOException {

	// restore the old graphics state
	printer.println("\ngrestore");

	//	printer.println("\nshowpage\n%%Trailer");
	printer.println("\n%%Trailer\n\n%%EOF");
    }


    /**
     * <p>
     * Sets the current postscript drawing color.
     * </p>
     *
     * @param color The color to set
     * @param alpha The transparency to apply to the color
     */
    protected void setColor(Color color, double alpha) {

	// if the current color is the same as what we're trying to set
	// it to, then don't bother outputting a new color

	if ( (color.equals(currentColor)) && (alpha == currentTransparency) ) {
	    return;
	}
	else {
	    currentColor = color;
	    currentTransparency = alpha;
	}

	// postscript doesn't really "do" transparency, so we'll fake it
	// by mixing the paint color with the current background color.
	// Note that this does not make layered objects transparent; it
	// only makes objects transparent relative to the current background.

	float[] fg = color.getRGBColorComponents(null);
	float[] bg = backgroundColor.getRGBColorComponents(null);
	double[] trans = new double[3];
	
	for (int i = 0; i < trans.length; i++) {
	    trans[i] = (alpha * fg[i]) + ((1 - alpha) * bg[i]);
	}

	printer.println(trans[0] + " " + trans[1] + " " +
			trans[2] + " setrgbcolor");

    }


    /**
     * <p>
     * Sets the current stroking color.
     * </p>
     *
     * @param newColor The new stroking color
     */
    public void setStrokeColor(Color newColor) {

	strokeColor = newColor;

    }


    /**
     * <p>
     * Sets the current filling color.
     * </p>
     *
     * @param newColor The new filling color
     */
    public void setFillColor(Color newColor) {

	fillColor = newColor;

    }


    /**
     * <p>
     * Sets the pen stroke width and type.
     * </p>
     *
     * @param stroke The new stroke
     */
    public void setStroke(Stroke stroke) {

	BasicStroke s = (BasicStroke)stroke;

	// 0 == hairline
	printer.println(s.getLineWidth() + " setlinewidth");

	// get the cap type
	int cap = s.getEndCap();
	int myCap = 0;

	switch (cap) {
            
	case BasicStroke.CAP_BUTT:
	    myCap = 0;
	    break;
		
	case BasicStroke.CAP_ROUND:
	    myCap = 1;
	    break;
		
	case BasicStroke.CAP_SQUARE:
	    myCap = 2;
	    break;
	}
	printer.println(myCap + " setlinecap");

	// get the line join
	int line = s.getLineJoin();
	int myLine = 0;

	switch (line) {
            
	case BasicStroke.JOIN_MITER:
	    myLine = 0;
	    break;
	
	case BasicStroke.JOIN_ROUND:
	    myLine = 1;
	    break;
		
	case BasicStroke.JOIN_BEVEL:
	    myLine = 2;
	    break;
	}
	printer.println(myLine + " setlinejoin");

	// get miter limit
	printer.println(s.getMiterLimit() + " setmiterlimit");

	// get the dash
	float[] dash = s.getDashArray();

	if (dash != null) {
	    printer.print("[");
	    for (int i = 0; i < dash.length; i++) {
		printer.print(dash[i] + " ");
	    }
	    printer.println("] " + s.getDashPhase() + " setdash");
	}
	else {
	    printer.println("[] 0 setdash");
	}
    }


    /**
     * <p>
     * Sets the stroke transparency (between 0 and 1).
     * </p>
     *
     * @param transparency The new transparency of the pen
     */
    public void setStrokeTransparency(double transparency) {

	strokeTransparency = transparency;
    }


    /**
     * <p>
     * Sets the fill transparency (between 0 and 1).
     * </p>
     *
     * @param transparency The new transparency of the pen
     */
    public void setFillTransparency(double transparency) {

	fillTransparency = transparency;
    }


    /**
     * <p>
     * Draws a shape to the output file.
     * </p>
     *
     * @param shape The shape to draw
     */
    protected void outputPath(Shape shape) throws IOException {

	// open a new path
	printer.println("newpath");

	// get a path iterator over the shape
	PathIterator pi = shape.getPathIterator(transform);

	// Get the winding rule, and set the fill type as needed
	int winding = pi.getWindingRule();

	if (winding == PathIterator.WIND_EVEN_ODD) {
	    F = "eofill";
	}
	else {
	    // assume nonzero winding rule
	    F = "fill";
	}

	// now iterate over every path in the shape
	while (!pi.isDone()) {
	    
	    // construct an array to hold the coordinates of this segment
	    double[] points = new double[6];
	    // integer to hold the segment type
	    int segType;

	    segType = pi.currentSegment(points);

	    // switch on the segment type
	    switch (segType) {
            
	    case PathIterator.SEG_MOVETO:
		printer.println(truncate(points[0]) + " " +
				truncate(points[1]) + " moveto");
		break;
		
	    case PathIterator.SEG_LINETO:
		printer.println(truncate(points[0]) + " " +
				truncate(points[1]) + " lineto");
		break;
		
	    case PathIterator.SEG_CUBICTO:
		printer.println(truncate(points[0]) + " " +
				truncate(points[1]) + " " +
				truncate(points[2]) + " " +
				truncate(points[3]) + " " +
				truncate(points[4]) + " " +
				truncate(points[5]) + " curveto");
		break;
		
	    case PathIterator.SEG_QUADTO:
		System.err.println("Quadric Curve " +
				   points[0] + " " + points[1] + " " +
				   points[2] + " " + points[3] +
				   " NOT IMPLEMENTED");
		break;
		
	    case PathIterator.SEG_CLOSE:
		// nada
		break;
		
	    }

	    pi.next();

	}

	// close the path
	printer.println("closepath");

    }


    /**
     * <p>
     * Strokes a shape to the output file.
     * </p>
     *
     * @param shape The shape to stroke
     */
    public void strokeShape(Shape shape) throws IOException {

	setColor(strokeColor, strokeTransparency);
	outputPath(shape);
	printer.println(S);

    }


    /**
     * <p>
     * Fills a shape to the output file.
     * </p>
     *
     * @param shape The shape to fill
     */
    public void fillShape(Shape shape) throws IOException {

	setColor(fillColor, fillTransparency);
	outputPath(shape);
	printer.println(F);

    }


    /**
     * <p>
     * Strokes and fills the named template shapes to the output file, 
     * using the transforms provided to place them in the correct positions.
     * </p>
     *
     * @param name The name of the template to use
     * @param transforms The transformation to apply to the template shape
     */
    public void drawTemplates(String name, AffineTransform[] transforms)
	throws IOException {

	// this method is overridden to provide a more intelligent approach.
	// since postscript can't fill and stroke at the same time, it's best
	// to do all our filling followed by all our stroking.

	for (int i = 0; i < transforms.length; i++) {
	    fillTemplate(name, transforms[i]);
	}
	for (int j = 0; j < transforms.length; j++) {
	    strokeTemplate(name, transforms[j]);
	}
    }


    public static void main(String[] args) {

	File test = new File(args[0]);

	try {

	    Postscript ps = new Postscript(test, null, 612.0, 792.0,
					    100.0, 100.0, 400.0, 400.0,
					    "Test of PS code",
					    "Jason Healy",
					    "This is a test",
					    "test ps",
					    "The main() method of the PS code");
	    
	    ps.prepare();

	    ps.setStroke(new BasicStroke());

	    ps.setStrokeColor(Color.black);
	    ps.strokeShape(new java.awt.geom.Rectangle2D.Double(110,110,280,280));

	    ps.setFillColor(Color.blue);
	    ps.fillShape(new java.awt.geom.Rectangle2D.Double(150,150,100,200));

	    ps.drawShape(new java.awt.geom.Ellipse2D.Double(120,120,200,100));

	    ps.finish();

	}
	catch (IOException ioe) {
	    System.err.println("Caught IO Exception" + ioe.getMessage());
	    ioe.printStackTrace();
	}
    }

}
