// $Header: /opt/cvs/java/net/logn/util/vectoroutput/VectorOutput.java,v 1.6 2001/02/15 14:18:24 jhealy Exp $
// Copyright 2001 Jason Healy.  Please see file COPYRIGHT for details.

package net.logn.util.vectoroutput;

// AWT Drawing classes
import java.awt.Shape;
import java.awt.Color;
import java.awt.Stroke;
// AffineTransforms for mapping
import java.awt.geom.AffineTransform;
// File writing capabilities
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;


/**
 * <p>
 * <b>VectorOutput</b> defines the interface for classes that take vector
 * shapes and then render them to a file format, such as PDF of PNG.
 * </p>
 *
 * @author Jason Healy
 * @version $Revision: 1.6 $
 *
 * Last Modified $Date: 2001/02/15 14:18:24 $ by $Author: jhealy $
 */
public interface VectorOutput {


    /**
     * <p>
     * Prepares the file by writing any necessary header information.
     * </p>
     *
     */
    public void prepare() throws IOException;


    /**
     * <p>
     * Finishes writing the file and closes the output file stream.
     * </p>
     *
     */
    public void finish() throws IOException;


    /**
     * <p>
     * Sets the current stroking color.
     * </p>
     *
     * @param newColor The new stroking color
     */
    public void setStrokeColor(Color newColor) throws IOException;


    /**
     * <p>
     * Sets the pen stroke width and type
     * </p>
     *
     * @param stroke The new stroke
     */
    public void setStroke(Stroke stroke) throws IOException;


    /**
     * <p>
     * Sets the pen transparency (between 0 and 1)
     * </p>
     *
     * @param transparency The new transparency of the pen
     */
    public void setStrokeTransparency(double transparency) throws IOException;


    /**
     * <p>
     * Sets the current filling color.
     * </p>
     *
     * @param newColor The new filling color
     */
    public void setFillColor(Color newColor) throws IOException;


    /**
     * <p>
     * Sets the fill transparency (between 0 and 1)
     * </p>
     *
     * @param transparency The new transparency of the pen
     */
    public void setFillTransparency(double transparency) throws IOException;


    /**
     * <p>
     * Strokes a shape to the output file.
     * </p>
     *
     * @param shape The shape to stroke
     */
    public void strokeShape(Shape shape) throws IOException;


    /**
     * <p>
     * Strokes all the provided shapes to the output file.
     * </p>
     *
     * @param shapes The shapes to stroke
     */
    public void strokeShapes(Shape[] shapes) throws IOException;


    /**
     * <p>
     * Fills a shape to the output file.
     * </p>
     *
     * @param shape The shape to fill
     */
    public void fillShape(Shape shape) throws IOException;


    /**
     * <p>
     * Fills all the provided shapes to the output file.
     * </p>
     *
     * @param shapes The shapes to fill
     */
    public void fillShapes(Shape[] shapes) throws IOException;


    /**
     * <p>
     * Strokes and fills a shape to the output file.
     * </p>
     *
     * @param shape The shape to stroke and fill
     */
    public void drawShape(Shape shape) throws IOException;


    /**
     * <p>
     * Strokes and fills all the shapes to the output file.
     * </p>
     *
     * @param shapes The shapes to stroke and fill
     */
    public void drawShapes(Shape[] shapes) throws IOException;


    /**
     * <p>
     * Creates a "template shape" that will be repeated often in the
     * image.  Subsequent calls can be made to the drawTemplate() and
     * fillTemplate() methods to fill these predefined shapes.
     * </p>
     *
     * @param name The name of the template
     * @param shape The shape to use for the template
     */
    public void createTemplate(String name, Shape shape) throws IOException;


    /**
     * <p>
     * Strokes the named template shape to the output file, using the 
     * transform provided to place it in the correct position.
     * </p>
     *
     * @param name The name of the template to use
     * @param transform The transformation to apply to the template shape
     */
    public void strokeTemplate(String name, AffineTransform transform)
	throws IOException;


    /**
     * <p>
     * Strokes the named template shape to the output file, using the
     * transforms provided to place them in the correct positions.
     * </p>
     *
     * @param name The name of the template to use
     * @param transforms The transformation to apply to the template shape
     */
    public void strokeTemplates(String name, AffineTransform[] transforms)
	throws IOException;


    /**
     * <p>
     * Fills the named template shape to the output file, using the transform
     * provided to place it in the correct position.
     * </p>
     *
     * @param name The name of the template to use
     * @param transform The transformation to apply to the template shape
     */
    public void fillTemplate(String name, AffineTransform transform)
	throws IOException;
    

    /**
     * <p>
     * Fills the named template shape to the output file, using the transforms
     * provided to place them in the correct positions.
     * </p>
     *
     * @param name The name of the template to use
     * @param transforms The transformation to apply to the template shape
     */
    public void fillTemplates(String name, AffineTransform[] transforms)
	throws IOException;


    /**
     * <p>
     * Strokes and fills the named template shape to the output file, 
     * using the transform provided to place it in the correct position.
     * </p>
     *
     * @param name The name of the template to use
     * @param transform The transformation to apply to the template shape
     */
    public void drawTemplate(String name, AffineTransform transform)
	throws IOException;


    /**
     * <p>
     * Strokes and fills the named template shape to the output file, 
     * using the transforms provided to place them in the correct positions.
     * </p>
     *
     * @param name The name of the template to use
     * @param transforms The transformation to apply to the template shape
     */
    public void drawTemplates(String name, AffineTransform[] transforms)
	throws IOException;


}
